"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Nfsv4Encoder = void 0;
const tslib_1 = require("tslib");
const Writer_1 = require("@jsonjoy.com/util/lib/buffers/Writer");
const XdrEncoder_1 = require("../../xdr/XdrEncoder");
const errors_1 = require("./errors");
const msg = tslib_1.__importStar(require("./messages"));
const structs = tslib_1.__importStar(require("./structs"));
class Nfsv4Encoder {
    constructor(writer = new Writer_1.Writer()) {
        this.writer = writer;
        this.xdr = new XdrEncoder_1.XdrEncoder(writer);
    }
    encodeCompound(compound, isRequest) {
        if (isRequest)
            this.writeCompoundRequest(compound);
        else
            this.writeCompoundResponse(compound);
        return this.writer.flush();
    }
    writeCompound(compound, isRequest) {
        if (isRequest)
            this.writeCompoundRequest(compound);
        else
            this.writeCompoundResponse(compound);
    }
    writeCompoundRequest(request) {
        const xdr = this.xdr;
        xdr.writeStr(request.tag);
        xdr.writeUnsignedInt(request.minorversion);
        xdr.writeUnsignedInt(request.argarray.length);
        for (const arg of request.argarray) {
            this.writeRequest(arg);
        }
    }
    writeCompoundResponse(response) {
        const xdr = this.xdr;
        xdr.writeUnsignedInt(response.status);
        xdr.writeStr(response.tag);
        xdr.writeUnsignedInt(response.resarray.length);
        for (const res of response.resarray) {
            this.writeResponse(res);
        }
    }
    writeRequest(request) {
        if (request instanceof msg.Nfsv4AccessRequest) {
            this.xdr.writeUnsignedInt(3);
            this.writeAccessRequest(request);
        }
        else if (request instanceof msg.Nfsv4CloseRequest) {
            this.xdr.writeUnsignedInt(4);
            this.writeCloseRequest(request);
        }
        else if (request instanceof msg.Nfsv4CommitRequest) {
            this.xdr.writeUnsignedInt(5);
            this.writeCommitRequest(request);
        }
        else if (request instanceof msg.Nfsv4CreateRequest) {
            this.xdr.writeUnsignedInt(6);
            this.writeCreateRequest(request);
        }
        else if (request instanceof msg.Nfsv4DelegreturnRequest) {
            this.xdr.writeUnsignedInt(8);
            this.writeDelegreturnRequest(request);
        }
        else if (request instanceof msg.Nfsv4GetattrRequest) {
            this.xdr.writeUnsignedInt(9);
            this.writeGetattrRequest(request);
        }
        else if (request instanceof msg.Nfsv4GetfhRequest) {
            this.xdr.writeUnsignedInt(10);
            this.writeGetfhRequest(request);
        }
        else if (request instanceof msg.Nfsv4LinkRequest) {
            this.xdr.writeUnsignedInt(11);
            this.writeLinkRequest(request);
        }
        else if (request instanceof msg.Nfsv4LockRequest) {
            this.xdr.writeUnsignedInt(12);
            this.writeLockRequest(request);
        }
        else if (request instanceof msg.Nfsv4LocktRequest) {
            this.xdr.writeUnsignedInt(13);
            this.writeLocktRequest(request);
        }
        else if (request instanceof msg.Nfsv4LockuRequest) {
            this.xdr.writeUnsignedInt(14);
            this.writeLockuRequest(request);
        }
        else if (request instanceof msg.Nfsv4LookupRequest) {
            this.xdr.writeUnsignedInt(15);
            this.writeLookupRequest(request);
        }
        else if (request instanceof msg.Nfsv4LookuppRequest) {
            this.xdr.writeUnsignedInt(16);
            this.writeLookuppRequest(request);
        }
        else if (request instanceof msg.Nfsv4NverifyRequest) {
            this.xdr.writeUnsignedInt(17);
            this.writeNverifyRequest(request);
        }
        else if (request instanceof msg.Nfsv4OpenRequest) {
            this.xdr.writeUnsignedInt(18);
            this.writeOpenRequest(request);
        }
        else if (request instanceof msg.Nfsv4OpenConfirmRequest) {
            this.xdr.writeUnsignedInt(20);
            this.writeOpenConfirmRequest(request);
        }
        else if (request instanceof msg.Nfsv4OpenDowngradeRequest) {
            this.xdr.writeUnsignedInt(21);
            this.writeOpenDowngradeRequest(request);
        }
        else if (request instanceof msg.Nfsv4PutfhRequest) {
            this.xdr.writeUnsignedInt(22);
            this.writePutfhRequest(request);
        }
        else if (request instanceof msg.Nfsv4PutpubfhRequest) {
            this.xdr.writeUnsignedInt(23);
            this.writePutpubfhRequest(request);
        }
        else if (request instanceof msg.Nfsv4PutrootfhRequest) {
            this.xdr.writeUnsignedInt(24);
            this.writePutrootfhRequest(request);
        }
        else if (request instanceof msg.Nfsv4ReadRequest) {
            this.xdr.writeUnsignedInt(25);
            this.writeReadRequest(request);
        }
        else if (request instanceof msg.Nfsv4ReaddirRequest) {
            this.xdr.writeUnsignedInt(26);
            this.writeReaddirRequest(request);
        }
        else if (request instanceof msg.Nfsv4ReadlinkRequest) {
            this.xdr.writeUnsignedInt(27);
            this.writeReadlinkRequest(request);
        }
        else if (request instanceof msg.Nfsv4RemoveRequest) {
            this.xdr.writeUnsignedInt(28);
            this.writeRemoveRequest(request);
        }
        else if (request instanceof msg.Nfsv4RenameRequest) {
            this.xdr.writeUnsignedInt(29);
            this.writeRenameRequest(request);
        }
        else if (request instanceof msg.Nfsv4RenewRequest) {
            this.xdr.writeUnsignedInt(30);
            this.writeRenewRequest(request);
        }
        else if (request instanceof msg.Nfsv4RestorefhRequest) {
            this.xdr.writeUnsignedInt(31);
            this.writeRestorefhRequest(request);
        }
        else if (request instanceof msg.Nfsv4SavefhRequest) {
            this.xdr.writeUnsignedInt(32);
            this.writeSavefhRequest(request);
        }
        else if (request instanceof msg.Nfsv4SecinfoRequest) {
            this.xdr.writeUnsignedInt(33);
            this.writeSecinfoRequest(request);
        }
        else if (request instanceof msg.Nfsv4SetattrRequest) {
            this.xdr.writeUnsignedInt(34);
            this.writeSetattrRequest(request);
        }
        else if (request instanceof msg.Nfsv4SetclientidRequest) {
            this.xdr.writeUnsignedInt(35);
            this.writeSetclientidRequest(request);
        }
        else if (request instanceof msg.Nfsv4SetclientidConfirmRequest) {
            this.xdr.writeUnsignedInt(36);
            this.writeSetclientidConfirmRequest(request);
        }
        else if (request instanceof msg.Nfsv4VerifyRequest) {
            this.xdr.writeUnsignedInt(37);
            this.writeVerifyRequest(request);
        }
        else if (request instanceof msg.Nfsv4WriteRequest) {
            this.xdr.writeUnsignedInt(38);
            this.writeWriteRequest(request);
        }
        else {
            throw new errors_1.Nfsv4EncodingError(`Unknown request type: ${request.constructor.name}`);
        }
    }
    writeResponse(response) {
        if (response instanceof msg.Nfsv4AccessResponse) {
            this.xdr.writeUnsignedInt(3);
            this.writeAccessResponse(response);
        }
        else if (response instanceof msg.Nfsv4CloseResponse) {
            this.xdr.writeUnsignedInt(4);
            this.writeCloseResponse(response);
        }
        else if (response instanceof msg.Nfsv4CommitResponse) {
            this.xdr.writeUnsignedInt(5);
            this.writeCommitResponse(response);
        }
        else if (response instanceof msg.Nfsv4CreateResponse) {
            this.xdr.writeUnsignedInt(6);
            this.writeCreateResponse(response);
        }
        else if (response instanceof msg.Nfsv4DelegreturnResponse) {
            this.xdr.writeUnsignedInt(8);
            this.writeDelegreturnResponse(response);
        }
        else if (response instanceof msg.Nfsv4GetattrResponse) {
            this.xdr.writeUnsignedInt(9);
            this.writeGetattrResponse(response);
        }
        else if (response instanceof msg.Nfsv4GetfhResponse) {
            this.xdr.writeUnsignedInt(10);
            this.writeGetfhResponse(response);
        }
        else if (response instanceof msg.Nfsv4LinkResponse) {
            this.xdr.writeUnsignedInt(11);
            this.writeLinkResponse(response);
        }
        else if (response instanceof msg.Nfsv4LockResponse) {
            this.xdr.writeUnsignedInt(12);
            this.writeLockResponse(response);
        }
        else if (response instanceof msg.Nfsv4LocktResponse) {
            this.xdr.writeUnsignedInt(13);
            this.writeLocktResponse(response);
        }
        else if (response instanceof msg.Nfsv4LockuResponse) {
            this.xdr.writeUnsignedInt(14);
            this.writeLockuResponse(response);
        }
        else if (response instanceof msg.Nfsv4LookupResponse) {
            this.xdr.writeUnsignedInt(15);
            this.writeLookupResponse(response);
        }
        else if (response instanceof msg.Nfsv4LookuppResponse) {
            this.xdr.writeUnsignedInt(16);
            this.writeLookuppResponse(response);
        }
        else if (response instanceof msg.Nfsv4NverifyResponse) {
            this.xdr.writeUnsignedInt(17);
            this.writeNverifyResponse(response);
        }
        else if (response instanceof msg.Nfsv4OpenResponse) {
            this.xdr.writeUnsignedInt(18);
            this.writeOpenResponse(response);
        }
        else if (response instanceof msg.Nfsv4OpenConfirmResponse) {
            this.xdr.writeUnsignedInt(20);
            this.writeOpenConfirmResponse(response);
        }
        else if (response instanceof msg.Nfsv4OpenDowngradeResponse) {
            this.xdr.writeUnsignedInt(21);
            this.writeOpenDowngradeResponse(response);
        }
        else if (response instanceof msg.Nfsv4PutfhResponse) {
            this.xdr.writeUnsignedInt(22);
            this.writePutfhResponse(response);
        }
        else if (response instanceof msg.Nfsv4PutpubfhResponse) {
            this.xdr.writeUnsignedInt(23);
            this.writePutpubfhResponse(response);
        }
        else if (response instanceof msg.Nfsv4PutrootfhResponse) {
            this.xdr.writeUnsignedInt(24);
            this.writePutrootfhResponse(response);
        }
        else if (response instanceof msg.Nfsv4ReadResponse) {
            this.xdr.writeUnsignedInt(25);
            this.writeReadResponse(response);
        }
        else if (response instanceof msg.Nfsv4ReaddirResponse) {
            this.xdr.writeUnsignedInt(26);
            this.writeReaddirResponse(response);
        }
        else if (response instanceof msg.Nfsv4ReadlinkResponse) {
            this.xdr.writeUnsignedInt(27);
            this.writeReadlinkResponse(response);
        }
        else if (response instanceof msg.Nfsv4RemoveResponse) {
            this.xdr.writeUnsignedInt(28);
            this.writeRemoveResponse(response);
        }
        else if (response instanceof msg.Nfsv4RenameResponse) {
            this.xdr.writeUnsignedInt(29);
            this.writeRenameResponse(response);
        }
        else if (response instanceof msg.Nfsv4RenewResponse) {
            this.xdr.writeUnsignedInt(30);
            this.writeRenewResponse(response);
        }
        else if (response instanceof msg.Nfsv4RestorefhResponse) {
            this.xdr.writeUnsignedInt(31);
            this.writeRestorefhResponse(response);
        }
        else if (response instanceof msg.Nfsv4SavefhResponse) {
            this.xdr.writeUnsignedInt(32);
            this.writeSavefhResponse(response);
        }
        else if (response instanceof msg.Nfsv4SecinfoResponse) {
            this.xdr.writeUnsignedInt(33);
            this.writeSecinfoResponse(response);
        }
        else if (response instanceof msg.Nfsv4SetattrResponse) {
            this.xdr.writeUnsignedInt(34);
            this.writeSetattrResponse(response);
        }
        else if (response instanceof msg.Nfsv4SetclientidResponse) {
            this.xdr.writeUnsignedInt(35);
            this.writeSetclientidResponse(response);
        }
        else if (response instanceof msg.Nfsv4SetclientidConfirmResponse) {
            this.xdr.writeUnsignedInt(36);
            this.writeSetclientidConfirmResponse(response);
        }
        else if (response instanceof msg.Nfsv4VerifyResponse) {
            this.xdr.writeUnsignedInt(37);
            this.writeVerifyResponse(response);
        }
        else if (response instanceof msg.Nfsv4WriteResponse) {
            this.xdr.writeUnsignedInt(38);
            this.writeWriteResponse(response);
        }
        else {
            throw new errors_1.Nfsv4EncodingError(`Unknown response type: ${response.constructor.name}`);
        }
    }
    writeFh(fh) {
        this.xdr.writeVarlenOpaque(fh.data);
    }
    writeVerifier(verifier) {
        this.xdr.writeOpaque(verifier.data);
    }
    writeTime(time) {
        const xdr = this.xdr;
        xdr.writeHyper(time.seconds);
        xdr.writeUnsignedInt(time.nseconds);
    }
    writeStateid(stateid) {
        const xdr = this.xdr;
        xdr.writeUnsignedInt(stateid.seqid);
        xdr.writeOpaque(stateid.other);
    }
    writeBitmap(bitmap) {
        const xdr = this.xdr;
        xdr.writeUnsignedInt(bitmap.mask.length);
        for (const m of bitmap.mask) {
            xdr.writeUnsignedInt(m);
        }
    }
    writeFattr(fattr) {
        this.writeBitmap(fattr.attrmask);
        this.xdr.writeVarlenOpaque(fattr.attrVals);
    }
    writeChangeInfo(changeInfo) {
        const xdr = this.xdr;
        xdr.writeBoolean(changeInfo.atomic);
        xdr.writeUnsignedHyper(changeInfo.before);
        xdr.writeUnsignedHyper(changeInfo.after);
    }
    writeClientAddr(clientAddr) {
        const xdr = this.xdr;
        xdr.writeStr(clientAddr.rNetid);
        xdr.writeStr(clientAddr.rAddr);
    }
    writeCbClient(cbClient) {
        const xdr = this.xdr;
        xdr.writeUnsignedInt(cbClient.cbProgram);
        this.writeClientAddr(cbClient.cbLocation);
    }
    writeClientId(clientId) {
        this.writeVerifier(clientId.verifier);
        this.xdr.writeVarlenOpaque(clientId.id);
    }
    writeOpenOwner(openOwner) {
        const xdr = this.xdr;
        xdr.writeUnsignedHyper(openOwner.clientid);
        xdr.writeVarlenOpaque(openOwner.owner);
    }
    writeLockOwner(lockOwner) {
        const xdr = this.xdr;
        xdr.writeUnsignedHyper(lockOwner.clientid);
        xdr.writeVarlenOpaque(lockOwner.owner);
    }
    writeOpenToLockOwner(openToLockOwner) {
        const xdr = this.xdr;
        xdr.writeUnsignedInt(openToLockOwner.openSeqid);
        this.writeStateid(openToLockOwner.openStateid);
        xdr.writeUnsignedInt(openToLockOwner.lockSeqid);
        this.writeLockOwner(openToLockOwner.lockOwner);
    }
    writeExistingLockOwner(existingLockOwner) {
        this.writeStateid(existingLockOwner.lockStateid);
        this.xdr.writeUnsignedInt(existingLockOwner.lockSeqid);
    }
    writeOpenClaim(openClaim) {
        const xdr = this.xdr;
        xdr.writeUnsignedInt(openClaim.claimType);
        const claim = openClaim.claim;
        if (claim instanceof structs.Nfsv4OpenClaimNull) {
            xdr.writeStr(claim.file);
        }
        else if (claim instanceof structs.Nfsv4OpenClaimPrevious) {
            xdr.writeUnsignedInt(claim.delegateType);
        }
        else if (claim instanceof structs.Nfsv4OpenClaimDelegateCur) {
            this.writeStateid(claim.delegateStateid);
            xdr.writeStr(claim.file);
        }
        else if (claim instanceof structs.Nfsv4OpenClaimDelegatePrev) {
            xdr.writeStr(claim.file);
        }
    }
    writeAccessRequest(req) {
        this.xdr.writeUnsignedInt(req.access);
    }
    writeAccessResponse(res) {
        const xdr = this.xdr;
        xdr.writeUnsignedInt(res.status);
        if (res.status === 0 && res.resok) {
            xdr.writeUnsignedInt(res.resok.supported);
            xdr.writeUnsignedInt(res.resok.access);
        }
    }
    writeCloseRequest(req) {
        const xdr = this.xdr;
        xdr.writeUnsignedInt(req.seqid);
        this.writeStateid(req.openStateid);
    }
    writeCloseResponse(res) {
        const xdr = this.xdr;
        xdr.writeUnsignedInt(res.status);
        if (res.status === 0 && res.resok) {
            this.writeStateid(res.resok.openStateid);
        }
    }
    writeCommitRequest(req) {
        const xdr = this.xdr;
        xdr.writeUnsignedHyper(req.offset);
        xdr.writeUnsignedInt(req.count);
    }
    writeCommitResponse(res) {
        const xdr = this.xdr;
        xdr.writeUnsignedInt(res.status);
        if (res.status === 0 && res.resok) {
            this.writeVerifier(res.resok.writeverf);
        }
    }
    writeCreateRequest(req) {
        const xdr = this.xdr;
        const createType = req.objtype;
        xdr.writeUnsignedInt(createType.type);
        const objtype = createType.objtype;
        if (objtype instanceof structs.Nfsv4CreateTypeLink) {
            xdr.writeStr(objtype.linkdata);
        }
        else if (objtype instanceof structs.Nfsv4CreateTypeDevice) {
            xdr.writeUnsignedInt(objtype.devdata.specdata1);
            xdr.writeUnsignedInt(objtype.devdata.specdata2);
        }
        xdr.writeStr(req.objname);
        if (objtype instanceof structs.Nfsv4CreateTypeLink) {
            this.writeFattr(objtype.createattrs);
        }
        else if (objtype instanceof structs.Nfsv4CreateTypeDevice) {
            this.writeFattr(objtype.createattrs);
        }
        else if (objtype instanceof structs.Nfsv4CreateTypeOther) {
            this.writeFattr(objtype.createattrs);
        }
    }
    writeCreateResponse(res) {
        const xdr = this.xdr;
        xdr.writeUnsignedInt(res.status);
        if (res.status === 0 && res.resok) {
            this.writeChangeInfo(res.resok.cinfo);
            this.writeBitmap(res.resok.attrset);
        }
    }
    writeDelegreturnRequest(req) {
        this.writeStateid(req.delegStateid);
    }
    writeDelegreturnResponse(res) {
        this.xdr.writeUnsignedInt(res.status);
    }
    writeGetattrRequest(req) {
        this.writeBitmap(req.attrRequest);
    }
    writeGetattrResponse(res) {
        const xdr = this.xdr;
        xdr.writeUnsignedInt(res.status);
        if (res.status === 0 && res.resok) {
            this.writeFattr(res.resok.objAttributes);
        }
    }
    writeGetfhRequest(req) {
    }
    writeGetfhResponse(res) {
        const xdr = this.xdr;
        xdr.writeUnsignedInt(res.status);
        if (res.status === 0 && res.resok) {
            this.writeFh(res.resok.object);
        }
    }
    writeLinkRequest(req) {
        this.xdr.writeStr(req.newname);
    }
    writeLinkResponse(res) {
        const xdr = this.xdr;
        xdr.writeUnsignedInt(res.status);
        if (res.status === 0 && res.resok) {
            this.writeChangeInfo(res.resok.cinfo);
        }
    }
    writeLockRequest(req) {
        const xdr = this.xdr;
        xdr.writeUnsignedInt(req.locktype);
        xdr.writeBoolean(req.reclaim);
        xdr.writeUnsignedHyper(req.offset);
        xdr.writeUnsignedHyper(req.length);
        xdr.writeBoolean(req.locker.newLockOwner);
        const owner = req.locker.owner;
        if (owner instanceof structs.Nfsv4LockNewOwner) {
            this.writeOpenToLockOwner(owner.openToLockOwner);
        }
        else if (owner instanceof structs.Nfsv4LockExistingOwner) {
            this.writeExistingLockOwner(owner);
        }
    }
    writeLockResponse(res) {
        const xdr = this.xdr;
        xdr.writeUnsignedInt(res.status);
        if (res.status === 0 && res.resok) {
            this.writeStateid(res.resok.lockStateid);
        }
        else if (res.denied) {
            xdr.writeUnsignedHyper(res.denied.offset);
            xdr.writeUnsignedHyper(res.denied.length);
            xdr.writeUnsignedInt(res.denied.locktype);
            this.writeLockOwner(res.denied.owner);
        }
    }
    writeLocktRequest(req) {
        const xdr = this.xdr;
        xdr.writeUnsignedInt(req.locktype);
        xdr.writeUnsignedHyper(req.offset);
        xdr.writeUnsignedHyper(req.length);
        this.writeLockOwner(req.owner);
    }
    writeLocktResponse(res) {
        const xdr = this.xdr;
        xdr.writeUnsignedInt(res.status);
        if (res.denied) {
            xdr.writeUnsignedHyper(res.denied.offset);
            xdr.writeUnsignedHyper(res.denied.length);
            xdr.writeUnsignedInt(res.denied.locktype);
            this.writeLockOwner(res.denied.owner);
        }
    }
    writeLockuRequest(req) {
        const xdr = this.xdr;
        xdr.writeUnsignedInt(req.locktype);
        xdr.writeUnsignedInt(req.seqid);
        this.writeStateid(req.lockStateid);
        xdr.writeUnsignedHyper(req.offset);
        xdr.writeUnsignedHyper(req.length);
    }
    writeLockuResponse(res) {
        const xdr = this.xdr;
        xdr.writeUnsignedInt(res.status);
        if (res.status === 0 && res.resok) {
            this.writeStateid(res.resok.lockStateid);
        }
    }
    writeLookupRequest(req) {
        this.xdr.writeStr(req.objname);
    }
    writeLookupResponse(res) {
        this.xdr.writeUnsignedInt(res.status);
    }
    writeLookuppRequest(req) {
    }
    writeLookuppResponse(res) {
        this.xdr.writeUnsignedInt(res.status);
    }
    writeNverifyRequest(req) {
        this.writeFattr(req.objAttributes);
    }
    writeNverifyResponse(res) {
        this.xdr.writeUnsignedInt(res.status);
    }
    writeOpenRequest(req) {
        const xdr = this.xdr;
        xdr.writeUnsignedInt(req.seqid);
        xdr.writeUnsignedInt(req.shareAccess);
        xdr.writeUnsignedInt(req.shareDeny);
        this.writeOpenOwner(req.owner);
        xdr.writeUnsignedInt(req.openhow);
        this.writeOpenClaim(req.claim);
    }
    writeOpenResponse(res) {
        const xdr = this.xdr;
        xdr.writeUnsignedInt(res.status);
        if (res.status === 0 && res.resok) {
            this.writeStateid(res.resok.stateid);
            this.writeChangeInfo(res.resok.cinfo);
            xdr.writeUnsignedInt(res.resok.rflags);
            this.writeBitmap(res.resok.attrset);
            this.writeOpenDelegation(res.resok.delegation);
        }
    }
    writeOpenDelegation(delegation) {
        const xdr = this.xdr;
        xdr.writeUnsignedInt(delegation.delegationType);
        if (delegation.delegation) {
            const deleg = delegation.delegation;
            if (deleg instanceof structs.Nfsv4OpenReadDelegation) {
                this.writeStateid(deleg.stateid);
                xdr.writeBoolean(deleg.recall);
                xdr.writeUnsignedInt(deleg.permissions.length);
                for (const ace of deleg.permissions) {
                    this.writeAce(ace);
                }
            }
            else if (deleg instanceof structs.Nfsv4OpenWriteDelegation) {
                this.writeStateid(deleg.stateid);
                xdr.writeBoolean(deleg.recall);
                xdr.writeUnsignedHyper(deleg.spaceLimit);
                xdr.writeUnsignedInt(deleg.permissions.length);
                for (const ace of deleg.permissions) {
                    this.writeAce(ace);
                }
            }
        }
    }
    writeAce(ace) {
        const xdr = this.xdr;
        xdr.writeUnsignedInt(ace.type);
        xdr.writeUnsignedInt(ace.flag);
        xdr.writeUnsignedInt(ace.accessMask);
        xdr.writeStr(ace.who);
    }
    writeOpenConfirmRequest(req) {
        this.writeStateid(req.openStateid);
        this.xdr.writeUnsignedInt(req.seqid);
    }
    writeOpenConfirmResponse(res) {
        const xdr = this.xdr;
        xdr.writeUnsignedInt(res.status);
        if (res.status === 0 && res.resok) {
            this.writeStateid(res.resok.openStateid);
        }
    }
    writeOpenDowngradeRequest(req) {
        const xdr = this.xdr;
        this.writeStateid(req.openStateid);
        xdr.writeUnsignedInt(req.seqid);
        xdr.writeUnsignedInt(req.shareAccess);
        xdr.writeUnsignedInt(req.shareDeny);
    }
    writeOpenDowngradeResponse(res) {
        const xdr = this.xdr;
        xdr.writeUnsignedInt(res.status);
        if (res.status === 0 && res.resok) {
            this.writeStateid(res.resok.openStateid);
        }
    }
    writePutfhRequest(req) {
        this.writeFh(req.object);
    }
    writePutfhResponse(res) {
        this.xdr.writeUnsignedInt(res.status);
    }
    writePutpubfhRequest(req) {
    }
    writePutpubfhResponse(res) {
        this.xdr.writeUnsignedInt(res.status);
    }
    writePutrootfhRequest(req) {
    }
    writePutrootfhResponse(res) {
        this.xdr.writeUnsignedInt(res.status);
    }
    writeReadRequest(req) {
        const xdr = this.xdr;
        this.writeStateid(req.stateid);
        xdr.writeUnsignedHyper(req.offset);
        xdr.writeUnsignedInt(req.count);
    }
    writeReadResponse(res) {
        const xdr = this.xdr;
        xdr.writeUnsignedInt(res.status);
        if (res.status === 0 && res.resok) {
            xdr.writeBoolean(res.resok.eof);
            xdr.writeVarlenOpaque(res.resok.data);
        }
    }
    writeReaddirRequest(req) {
        const xdr = this.xdr;
        xdr.writeUnsignedHyper(req.cookie);
        this.writeVerifier(req.cookieverf);
        xdr.writeUnsignedInt(req.dircount);
        xdr.writeUnsignedInt(req.maxcount);
        this.writeBitmap(req.attrRequest);
    }
    writeReaddirResponse(res) {
        const xdr = this.xdr;
        xdr.writeUnsignedInt(res.status);
        if (res.status === 0 && res.resok) {
            this.writeVerifier(res.resok.cookieverf);
            for (const entry of res.resok.entries) {
                xdr.writeBoolean(true);
                xdr.writeUnsignedHyper(entry.cookie);
                xdr.writeStr(entry.name);
                this.writeFattr(entry.attrs);
            }
            xdr.writeBoolean(false);
            xdr.writeBoolean(res.resok.eof);
        }
    }
    writeReadlinkRequest(req) {
    }
    writeReadlinkResponse(res) {
        const xdr = this.xdr;
        xdr.writeUnsignedInt(res.status);
        if (res.status === 0 && res.resok) {
            xdr.writeStr(res.resok.link);
        }
    }
    writeRemoveRequest(req) {
        this.xdr.writeStr(req.target);
    }
    writeRemoveResponse(res) {
        const xdr = this.xdr;
        xdr.writeUnsignedInt(res.status);
        if (res.status === 0 && res.resok) {
            this.writeChangeInfo(res.resok.cinfo);
        }
    }
    writeRenameRequest(req) {
        const xdr = this.xdr;
        xdr.writeStr(req.oldname);
        xdr.writeStr(req.newname);
    }
    writeRenameResponse(res) {
        const xdr = this.xdr;
        xdr.writeUnsignedInt(res.status);
        if (res.status === 0 && res.resok) {
            this.writeChangeInfo(res.resok.sourceCinfo);
            this.writeChangeInfo(res.resok.targetCinfo);
        }
    }
    writeRenewRequest(req) {
        this.xdr.writeUnsignedHyper(req.clientid);
    }
    writeRenewResponse(res) {
        this.xdr.writeUnsignedInt(res.status);
    }
    writeRestorefhRequest(req) {
    }
    writeRestorefhResponse(res) {
        this.xdr.writeUnsignedInt(res.status);
    }
    writeSavefhRequest(req) {
    }
    writeSavefhResponse(res) {
        this.xdr.writeUnsignedInt(res.status);
    }
    writeSecinfoRequest(req) {
        this.xdr.writeStr(req.name);
    }
    writeSecinfoResponse(res) {
        const xdr = this.xdr;
        xdr.writeUnsignedInt(res.status);
        if (res.status === 0 && res.resok) {
            xdr.writeUnsignedInt(res.resok.flavors.length);
            for (const flavor of res.resok.flavors) {
                xdr.writeUnsignedInt(flavor.flavor);
                if (flavor.flavorInfo) {
                    xdr.writeVarlenOpaque(flavor.flavorInfo.oid);
                    xdr.writeUnsignedInt(flavor.flavorInfo.qop);
                    xdr.writeUnsignedInt(flavor.flavorInfo.service);
                }
            }
        }
    }
    writeSetattrRequest(req) {
        this.writeStateid(req.stateid);
        this.writeFattr(req.objAttributes);
    }
    writeSetattrResponse(res) {
        const xdr = this.xdr;
        xdr.writeUnsignedInt(res.status);
        if (res.resok) {
            this.writeBitmap(res.resok.attrsset);
        }
    }
    writeSetclientidRequest(req) {
        this.writeClientId(req.client);
        this.writeCbClient(req.callback);
        this.xdr.writeUnsignedInt(req.callbackIdent);
    }
    writeSetclientidResponse(res) {
        const xdr = this.xdr;
        xdr.writeUnsignedInt(res.status);
        if (res.status === 0 && res.resok) {
            xdr.writeUnsignedHyper(res.resok.clientid);
            this.writeVerifier(res.resok.setclientidConfirm);
        }
    }
    writeSetclientidConfirmRequest(req) {
        const xdr = this.xdr;
        xdr.writeUnsignedHyper(req.clientid);
        this.writeVerifier(req.setclientidConfirm);
    }
    writeSetclientidConfirmResponse(res) {
        this.xdr.writeUnsignedInt(res.status);
    }
    writeVerifyRequest(req) {
        this.writeFattr(req.objAttributes);
    }
    writeVerifyResponse(res) {
        this.xdr.writeUnsignedInt(res.status);
    }
    writeWriteRequest(req) {
        const xdr = this.xdr;
        this.writeStateid(req.stateid);
        xdr.writeUnsignedHyper(req.offset);
        xdr.writeUnsignedInt(req.stable);
        xdr.writeVarlenOpaque(req.data);
    }
    writeWriteResponse(res) {
        const xdr = this.xdr;
        xdr.writeUnsignedInt(res.status);
        if (res.status === 0 && res.resok) {
            xdr.writeUnsignedInt(res.resok.count);
            xdr.writeUnsignedInt(res.resok.committed);
            this.writeVerifier(res.resok.writeverf);
        }
    }
}
exports.Nfsv4Encoder = Nfsv4Encoder;
//# sourceMappingURL=Nfsv4Encoder.js.map