"use strict";
/*
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the
 * License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
var query_string_utils_1 = require("./query_string_utils");
var MOCK_LOCATION_UNDEFINED = 'undefined';
/**
 * Useful in the context of writing tests.
 */
var MockLocationLike = /** @class */ (function () {
    function MockLocationLike() {
        this.hash = MOCK_LOCATION_UNDEFINED;
        this.host = MOCK_LOCATION_UNDEFINED;
        this.origin = MOCK_LOCATION_UNDEFINED;
        this.hostname = MOCK_LOCATION_UNDEFINED;
        this.pathname = MOCK_LOCATION_UNDEFINED;
        this.port = MOCK_LOCATION_UNDEFINED;
        this.protocol = MOCK_LOCATION_UNDEFINED;
        this.search = MOCK_LOCATION_UNDEFINED;
        this.assign = function () { };
    }
    MockLocationLike.prototype.setHash = function (hash) {
        this.hash = hash;
        return this;
    };
    MockLocationLike.prototype.setHost = function (host) {
        this.host = host;
        return this;
    };
    MockLocationLike.prototype.setOrigin = function (origin) {
        this.origin = origin;
        return this;
    };
    MockLocationLike.prototype.setHostname = function (hostname) {
        this.hostname = hostname;
        return this;
    };
    MockLocationLike.prototype.setPathname = function (pathname) {
        this.pathname = pathname;
        return this;
    };
    MockLocationLike.prototype.setPort = function (port) {
        this.port = port;
        return this;
    };
    MockLocationLike.prototype.setProtocol = function (protocol) {
        this.protocol = protocol;
        return this;
    };
    MockLocationLike.prototype.setSearch = function (search) {
        this.search = search;
        return this;
    };
    MockLocationLike.prototype.setAssign = function (assign) {
        this.assign = assign;
        return this;
    };
    return MockLocationLike;
}());
describe('Query String Parser Tests', function () {
    var locationLike = new MockLocationLike();
    var parser = new query_string_utils_1.BasicQueryStringUtils();
    it('Empty query string should not blow up.', function () {
        locationLike.setSearch('?');
        var result = parser.parse(locationLike);
        var keys = Object.keys(result);
        expect(result).toBeTruthy();
        expect(keys.length).toBe(0, 'No query parameters provided');
    });
    it('Should parse simple query strings.', function () {
        locationLike.setSearch(encodeURI('key1=value1&key2=value 2& key3= value 3'));
        var result = parser.parse(locationLike);
        var keys = Object.keys(result);
        expect(result).toBeTruthy();
        expect(keys.length).toBe(3, '3 Query parameters should be present');
        expect(keys[0]).toBe('key1');
        expect(keys[1]).toBe('key2');
        expect(keys[2]).toBe(' key3');
        expect(result[keys[0]]).toBe('value1', 'Expected value is "value1"');
        expect(result[keys[1]]).toBe('value 2', 'Expected value is "value 2"');
        expect(result[keys[2]]).toBe(' value 3', 'Expected value is " value 3"');
    });
    it('Should handle params with no values', function () {
        locationLike.setSearch(encodeURI('key1=value1&key2='));
        var result = parser.parse(locationLike);
        var keys = Object.keys(result);
        expect(result).toBeTruthy();
        expect(keys.length).toBe(1, '1 Query parameter should be present');
        expect(keys[0]).toBe('key1');
        expect(result[keys[0]]).toBe('value1', 'Expected value is "value1"');
    });
    it('Should handle duplicate parameter values', function () {
        locationLike.setSearch(encodeURI('key1=value1&key1=value2'));
        var result = parser.parse(locationLike);
        var keys = Object.keys(result);
        expect(result).toBeTruthy();
        expect(keys.length).toBe(1, '1 Query parameter should be present');
        expect(keys[0]).toBe('key1');
        expect(result[keys[0]]).toBe('value2', 'Expected value is "value2"');
    });
    it('Should be able to deal with escaped # or ? characters', function () {
        locationLike.setSearch(encodeURI('key1=value1?&key2=value2 #'));
        var result = parser.parse(locationLike);
        var keys = Object.keys(result);
        expect(result).toBeTruthy();
        expect(keys.length).toBe(2, '2 Query parameters should be present');
        expect(keys[0]).toBe('key1');
        expect(keys[1]).toBe('key2');
        expect(result[keys[0]]).toBe('value1?', 'Expected value is "value1?"');
        expect(result[keys[1]]).toBe('value2 #', 'Expected value is "value2 #"');
    });
});
//# sourceMappingURL=data:application/json;base64,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