"use strict";
/*
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the
 * License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.App = void 0;
// Represents the test web app that uses the AppAuthJS library.
var authorization_request_1 = require("../authorization_request");
var authorization_request_handler_1 = require("../authorization_request_handler");
var authorization_service_configuration_1 = require("../authorization_service_configuration");
var logger_1 = require("../logger");
var redirect_based_handler_1 = require("../redirect_based_handler");
var token_request_1 = require("../token_request");
var token_request_handler_1 = require("../token_request_handler");
/* an example open id connect provider */
var openIdConnectUrl = 'https://accounts.google.com';
/* example client configuration */
var clientId = '511828570984-7nmej36h9j2tebiqmpqh835naet4vci4.apps.googleusercontent.com';
var redirectUri = 'http://localhost:8000/app/redirect.html';
var scope = 'openid';
/**
 * The Test application.
 */
var App = /** @class */ (function () {
    function App(snackbar) {
        var _this = this;
        this.snackbar = snackbar;
        this.notifier = new authorization_request_handler_1.AuthorizationNotifier();
        this.authorizationHandler = new redirect_based_handler_1.RedirectRequestHandler();
        this.tokenHandler = new token_request_handler_1.BaseTokenRequestHandler();
        // set notifier to deliver responses
        this.authorizationHandler.setAuthorizationNotifier(this.notifier);
        // set a listener to listen for authorization responses
        this.notifier.setAuthorizationListener(function (request, response, error) {
            (0, logger_1.log)('Authorization request complete ', request, response, error);
            if (response) {
                _this.request = request;
                _this.response = response;
                _this.code = response.code;
                _this.showMessage("Authorization Code ".concat(response.code));
            }
        });
    }
    App.prototype.showMessage = function (message) {
        var snackbar = this.snackbar['MaterialSnackbar'];
        snackbar.showSnackbar({ message: message });
    };
    App.prototype.fetchServiceConfiguration = function () {
        var _this = this;
        authorization_service_configuration_1.AuthorizationServiceConfiguration.fetchFromIssuer(openIdConnectUrl)
            .then(function (response) {
            (0, logger_1.log)('Fetched service configuration', response);
            _this.configuration = response;
            _this.showMessage('Completed fetching configuration');
        })
            .catch(function (error) {
            (0, logger_1.log)('Something bad happened', error);
            _this.showMessage("Something bad happened ".concat(error));
        });
    };
    App.prototype.makeAuthorizationRequest = function () {
        // create a request
        var request = new authorization_request_1.AuthorizationRequest({
            client_id: clientId,
            redirect_uri: redirectUri,
            scope: scope,
            response_type: authorization_request_1.AuthorizationRequest.RESPONSE_TYPE_CODE,
            state: undefined,
            extras: { 'prompt': 'consent', 'access_type': 'offline' }
        });
        if (this.configuration) {
            this.authorizationHandler.performAuthorizationRequest(this.configuration, request);
        }
        else {
            this.showMessage('Fetch Authorization Service configuration, before you make the authorization request.');
        }
    };
    App.prototype.makeTokenRequest = function () {
        var _this = this;
        if (!this.configuration) {
            this.showMessage('Please fetch service configuration.');
            return;
        }
        var request = null;
        if (this.code) {
            var extras = undefined;
            if (this.request && this.request.internal) {
                extras = {};
                extras['code_verifier'] = this.request.internal['code_verifier'];
            }
            // use the code to make the token request.
            request = new token_request_1.TokenRequest({
                client_id: clientId,
                redirect_uri: redirectUri,
                grant_type: token_request_1.GRANT_TYPE_AUTHORIZATION_CODE,
                code: this.code,
                refresh_token: undefined,
                extras: extras
            });
        }
        else if (this.tokenResponse) {
            // use the token response to make a request for an access token
            request = new token_request_1.TokenRequest({
                client_id: clientId,
                redirect_uri: redirectUri,
                grant_type: token_request_1.GRANT_TYPE_REFRESH_TOKEN,
                code: undefined,
                refresh_token: this.tokenResponse.refreshToken,
                extras: undefined
            });
        }
        if (request) {
            this.tokenHandler.performTokenRequest(this.configuration, request)
                .then(function (response) {
                var isFirstRequest = false;
                if (_this.tokenResponse) {
                    // copy over new fields
                    _this.tokenResponse.accessToken = response.accessToken;
                    _this.tokenResponse.issuedAt = response.issuedAt;
                    _this.tokenResponse.expiresIn = response.expiresIn;
                    _this.tokenResponse.tokenType = response.tokenType;
                    _this.tokenResponse.scope = response.scope;
                }
                else {
                    isFirstRequest = true;
                    _this.tokenResponse = response;
                }
                // unset code, so we can do refresh token exchanges subsequently
                _this.code = undefined;
                if (isFirstRequest) {
                    _this.showMessage("Obtained a refresh token ".concat(response.refreshToken));
                }
                else {
                    _this.showMessage("Obtained an access token ".concat(response.accessToken, "."));
                }
            })
                .catch(function (error) {
                (0, logger_1.log)('Something bad happened', error);
                _this.showMessage("Something bad happened ".concat(error));
            });
        }
    };
    App.prototype.checkForAuthorizationResponse = function () {
        this.authorizationHandler.completeAuthorizationRequestIfPossible();
    };
    return App;
}());
exports.App = App;
// export App
window['App'] = App;
//# sourceMappingURL=data:application/json;base64,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