import PropTypes from 'prop-types';
import { SIZES } from './constants';
import { isFunction, warn } from './utils';
const INPUT_PROPS_BLACKLIST = [
    { alt: 'onBlur', prop: 'onBlur' },
    { alt: 'onInputChange', prop: 'onChange' },
    { alt: 'onFocus', prop: 'onFocus' },
    { alt: 'onKeyDown', prop: 'onKeyDown' },
];
export const sizeType = PropTypes.oneOf(SIZES);
export function checkPropType(validator, callback) {
    return (props, propName, componentName) => {
        PropTypes.checkPropTypes({ [propName]: validator }, props, 'prop', componentName);
        isFunction(callback) && callback(props, propName, componentName);
    };
}
export function caseSensitiveType(props) {
    const { caseSensitive, filterBy } = props;
    warn(!caseSensitive || typeof filterBy !== 'function', 'Your `filterBy` function will override the `caseSensitive` prop.');
}
export function deprecated(validator, reason) {
    return (props, propName, componentName) => {
        if (props[propName] != null) {
            warn(false, `The \`${propName}\` prop is deprecated. ${reason}`);
        }
        return PropTypes.checkPropTypes({ [propName]: validator }, props, 'prop', componentName);
    };
}
export function defaultInputValueType(props) {
    const { defaultInputValue, defaultSelected, multiple, selected } = props;
    const name = defaultSelected.length ? 'defaultSelected' : 'selected';
    warn(!(!multiple &&
        defaultInputValue &&
        (defaultSelected.length || (selected && selected.length))), `\`defaultInputValue\` will be overridden by the value from \`${name}\`.`);
}
export function defaultSelectedType(props) {
    const { defaultSelected, multiple } = props;
    warn(multiple || defaultSelected.length <= 1, 'You are passing multiple options to the `defaultSelected` prop of a ' +
        'Typeahead in single-select mode. The selections will be truncated to a ' +
        'single selection.');
}
export function highlightOnlyResultType({ allowNew, highlightOnlyResult, }) {
    warn(!(highlightOnlyResult && allowNew), '`highlightOnlyResult` will not work with `allowNew`.');
}
export function ignoreDiacriticsType(props) {
    const { filterBy, ignoreDiacritics } = props;
    warn(ignoreDiacritics || typeof filterBy !== 'function', 'Your `filterBy` function will override the `ignoreDiacritics` prop.');
}
export function inputPropsType({ inputProps }) {
    if (!(inputProps &&
        Object.prototype.toString.call(inputProps) === '[object Object]')) {
        return;
    }
    INPUT_PROPS_BLACKLIST.forEach(({ alt, prop }) => {
        const msg = alt ? ` Use the top-level \`${alt}\` prop instead.` : null;
        warn(!inputProps[prop], `The \`${prop}\` property of \`inputProps\` will be ignored.${msg}`);
    });
}
export function isRequiredForA11y(props, propName, componentName) {
    warn(props[propName] != null, `The prop \`${propName}\` is required to make \`${componentName}\` ` +
        'accessible for users of assistive technologies such as screen readers.');
}
export function labelKeyType({ allowNew, labelKey }) {
    warn(!(isFunction(labelKey) && allowNew), '`labelKey` must be a string when `allowNew={true}`.');
}
export const optionType = PropTypes.oneOfType([
    PropTypes.object,
    PropTypes.string,
]);
export function selectedType({ multiple, onChange, selected }) {
    warn(multiple || !selected || selected.length <= 1, 'You are passing multiple options to the `selected` prop of a Typeahead ' +
        'in single-select mode. This may lead to unexpected behaviors or errors.');
    warn(!selected || (selected && isFunction(onChange)), 'You provided a `selected` prop without an `onChange` handler. If you ' +
        'want the typeahead to be uncontrolled, use `defaultSelected`. ' +
        'Otherwise, set `onChange`.');
}
