import ts from "typescript";
import type { Application } from "../application.js";
import { Comment, type CommentDisplayPart, type ContainerReflection, type DeclarationReflection, DocumentReflection, type ParameterReflection, ProjectReflection, Reflection, type ReflectionSymbolId, type SignatureReflection, type SomeType, type TypeParameterReflection } from "../models/index.js";
import { Context } from "./context.js";
import { AbstractComponent } from "../utils/component.js";
import { type GlobString, MinimalSourceFile } from "#utils";
import type { DocumentationEntryPoint } from "../utils/entry-point.js";
import { type CommentParserConfig } from "./comments/index.js";
import type { CommentStyle, ValidationOptions } from "../utils/options/declaration.js";
import { type ExternalResolveResult, type ExternalSymbolResolver } from "./comments/linkResolver.js";
import { type DeclarationReference } from "#utils";
import type { FileRegistry } from "../models/FileRegistry.js";
import { IncludePlugin } from "./plugins/IncludePlugin.js";
export interface ConverterEvents {
    begin: [Context];
    end: [Context];
    createProject: [Context, ProjectReflection];
    createDeclaration: [Context, DeclarationReflection];
    createDocument: [undefined, DocumentReflection];
    createSignature: [
        Context,
        SignatureReflection,
        (ts.SignatureDeclaration | ts.IndexSignatureDeclaration | ts.JSDocSignature)?,
        ts.Signature?
    ];
    createParameter: [Context, ParameterReflection, ts.Node?];
    createTypeParameter: [
        Context,
        TypeParameterReflection,
        ts.TypeParameterDeclaration?
    ];
    resolveBegin: [Context];
    resolveReflection: [Context, Reflection];
    resolveEnd: [Context];
}
/**
 * Compiles source files using TypeScript and converts compiler symbols to reflections.
 *
 * @group None
 * @summary Responsible for converting TypeScript symbols into {@link Reflection}s and {@link Type}s.
 */
export declare class Converter extends AbstractComponent<Application, ConverterEvents> {
    /** @internal */
    accessor externalPattern: GlobString[];
    private externalPatternCache?;
    private excludeCache?;
    /** @internal */
    accessor excludeExternals: boolean;
    /** @internal */
    accessor excludePrivate: boolean;
    /** @internal */
    accessor excludeProtected: boolean;
    /** @internal */
    accessor excludeReferences: boolean;
    /** @internal */
    accessor commentStyle: CommentStyle;
    /** @internal */
    accessor validation: ValidationOptions;
    /** @internal */
    accessor externalSymbolLinkMappings: Record<string, Record<string, string>>;
    /** @internal */
    accessor preserveLinkText: boolean;
    /** @internal */
    accessor maxTypeConversionDepth: number;
    private _config?;
    private _externalSymbolResolvers;
    private _deferPermitted;
    private _defer;
    get config(): CommentParserConfig;
    /**
     * General events
     */
    /**
     * Triggered when the converter begins converting a project.
     * The listener will be given a {@link Context} object.
     * @event
     */
    static readonly EVENT_BEGIN: "begin";
    /**
     * Triggered when the converter has finished converting a project.
     * The listener will be given a {@link Context} object.
     * @event
     */
    static readonly EVENT_END: "end";
    /**
     * Factory events
     */
    /**
     * Triggered when the converter has created a project reflection.
     * The listener will be given {@link Context} and a {@link Models.ProjectReflection}.
     * @event
     */
    static readonly EVENT_CREATE_PROJECT: "createProject";
    /**
     * Triggered when the converter has created a declaration reflection.
     * The listener will be given {@link Context} and a {@link Models.DeclarationReflection}.
     * @event
     */
    static readonly EVENT_CREATE_DECLARATION: "createDeclaration";
    /**
     * Triggered when the converter has created a document reflection.
     * The listener will be given `undefined` (for consistency with the
     * other create events) and a {@link Models.DocumentReflection}.
     * @event
     */
    static readonly EVENT_CREATE_DOCUMENT: "createDocument";
    /**
     * Triggered when the converter has created a signature reflection.
     * The listener will be given {@link Context}, {@link Models.SignatureReflection} | {@link Models.ProjectReflection} the declaration,
     * `ts.SignatureDeclaration | ts.IndexSignatureDeclaration | ts.JSDocSignature | undefined`,
     * and `ts.Signature | undefined`. The signature will be undefined if the created signature is an index signature.
     * @event
     */
    static readonly EVENT_CREATE_SIGNATURE: "createSignature";
    /**
     * Triggered when the converter has created a parameter reflection.
     * The listener will be given {@link Context}, {@link Models.ParameterReflection} and a `ts.Node?`
     * @event
     */
    static readonly EVENT_CREATE_PARAMETER: "createParameter";
    /**
     * Triggered when the converter has created a type parameter reflection.
     * The listener will be given {@link Context} and a {@link Models.TypeParameterReflection}
     * @event
     */
    static readonly EVENT_CREATE_TYPE_PARAMETER: "createTypeParameter";
    /**
     * Resolve events
     */
    /**
     * Triggered when the converter begins resolving a project.
     * The listener will be given {@link Context}.
     * @event
     */
    static readonly EVENT_RESOLVE_BEGIN: "resolveBegin";
    /**
     * Triggered when the converter resolves a reflection.
     * The listener will be given {@link Context} and a {@link Reflection}.
     * @event
     */
    static readonly EVENT_RESOLVE: "resolveReflection";
    /**
     * Triggered when the converter has finished resolving a project.
     * The listener will be given {@link Context}.
     * @event
     */
    static readonly EVENT_RESOLVE_END: "resolveEnd";
    /** @internal @hidden */
    includePlugin: IncludePlugin;
    constructor(owner: Application);
    /**
     * Compile the given source files and create a project reflection for them.
     */
    convert(entryPoints: readonly DocumentationEntryPoint[]): ProjectReflection;
    /** @internal */
    addProjectDocuments(project: ProjectReflection): void;
    /** @internal */
    convertSymbol(context: Context, symbol: ts.Symbol, exportSymbol?: ts.Symbol): void;
    /**
     * Convert the given TypeScript type into its TypeDoc type reflection.
     *
     * @param context  The context object describing the current state the converter is in.
     * @returns The TypeDoc type reflection representing the given node and type.
     * @internal
     */
    convertType(context: Context, node: ts.TypeNode | undefined): SomeType;
    convertType(context: Context, type: ts.Type, node?: ts.TypeNode): SomeType;
    /**
     * Parse the given file into a comment. Intended to be used with markdown files.
     */
    parseRawComment(file: MinimalSourceFile, files: FileRegistry): {
        content: CommentDisplayPart[];
        frontmatter: Record<string, unknown>;
    };
    /**
     * Adds a new resolver that the theme can use to try to figure out how to link to a symbol declared
     * by a third-party library which is not included in the documentation.
     *
     * The resolver function will be passed a declaration reference which it can attempt to resolve. If
     * resolution fails, the function should return undefined.
     *
     * Note: This will be used for both references to types declared in node_modules (in which case the
     * reference passed will have the `moduleSource` set and the `symbolReference` will navigate via `.`)
     * and user defined \{\@link\} tags which cannot be resolved. If the link being resolved is inferred
     * from a type, then no `part` will be passed to the resolver function.
     */
    addUnknownSymbolResolver(resolver: ExternalSymbolResolver): void;
    /** @internal */
    resolveExternalLink(ref: DeclarationReference, refl: Reflection, part: CommentDisplayPart | undefined, symbolId: ReflectionSymbolId | undefined): ExternalResolveResult | string | undefined;
    resolveLinks(reflection: Reflection): void;
    /** @deprecated just pass the reflection */
    resolveLinks(comment: Comment, owner: Reflection): void;
    resolveLinks(parts: readonly CommentDisplayPart[], owner: Reflection): CommentDisplayPart[];
    /**
     * Permit deferred conversion steps to take place. Until this is called, {@link deferConversion}
     * will throw if used.
     * @since 0.28.1
     */
    permitDeferredConversion(): void;
    /**
     * Finalize deferred conversion, must be called by the caller of {@link permitDeferredConversion}
     * @since 0.28.1
     */
    finalizeDeferredConversion(): void;
    /**
     * Defer a conversion step until later. This may only be called during conversion.
     * @since 0.28.0
     */
    deferConversion(cb: () => void): void;
    /**
     * Compile the files within the given context and convert the compiler symbols to reflections.
     *
     * @param context  The context object describing the current state the converter is in.
     * @returns An array containing all errors generated by the TypeScript compiler.
     */
    private compile;
    private convertExports;
    /**
     * Resolve the project within the given context.
     *
     * @param context  The context object describing the current state the converter is in.
     * @returns The final project reflection.
     */
    private resolve;
    /**
     * Used to determine if we should immediately bail when creating a reflection.
     * Note: This should not be used for excludeNotDocumented because we don't have enough
     * information at this point since comment discovery hasn't happened.
     * @internal
     */
    shouldIgnore(symbol: ts.Symbol, checker: ts.TypeChecker): boolean;
    private isExcluded;
    /** @internal */
    isExternal(symbol: ts.Symbol, checker: ts.TypeChecker): boolean;
    processDocumentTags(reflection: Reflection, parent: ContainerReflection): void;
    private addDocument;
    private _buildCommentParserConfig;
}
