import { Compilation } from 'webpack';
import { Compiler } from 'webpack';
import * as postcss from 'postcss';

/**
 * @public
 */
export declare type BlocklistFunction = () => StringRegExpArray;

/**
 * @public
 */
declare type ComplexSafelist = {
    standard?: StringRegExpArray;
    /**
     * You can safelist selectors and their children based on a regular
     * expression with `safelist.deep`
     *
     * @example
     *
     * ```ts
     * const purgecss = await new PurgeCSS().purge({
     *   content: [],
     *   css: [],
     *   safelist: {
     *     deep: [/red$/]
     *   }
     * })
     * ```
     *
     * In this example, selectors such as `.bg-red .child-of-bg` will be left
     * in the final CSS, even if `child-of-bg` is not found.
     *
     */
    deep?: RegExp[];
    greedy?: RegExp[];
    variables?: StringRegExpArray;
    keyframes?: StringRegExpArray;
};

/**
 * @public
 */
declare type ExtractorFunction<T = string> = (content: T) => ExtractorResult;

/**
 * @public
 */
declare type ExtractorResult = ExtractorResultDetailed | string[];

/**
 * @public
 */
declare interface ExtractorResultDetailed {
    attributes: {
        names: string[];
        values: string[];
    };
    classes: string[];
    ids: string[];
    tags: string[];
    undetermined: string[];
}

/**
 * @public
 */
declare interface Extractors {
    extensions: string[];
    extractor: ExtractorFunction;
}

/**
 * @public
 */
export declare type PathFunction = () => string[];

/**
 * @public
 */
export declare class PurgeCSSPlugin {
    options: UserDefinedOptions;
    purgedStats: PurgedStats;
    constructor(options: UserDefinedOptions);
    apply(compiler: Compiler): void;
    initializePlugin(compilation: Compilation): void;
    runPluginHook(compilation: Compilation, entryPaths: string[]): Promise<void>;
}

/**
 * @public
 */
export declare type PurgedStats = {
    [index: string]: string[];
};

/**
 * @public
 */
declare interface RawContent<T = string> {
    extension: string;
    raw: T;
}

/**
 * @public
 */
declare interface RawCSS {
    raw: string;
    name?: string;
}

/**
 * @public
 */
export declare type SafelistFunction = () => ComplexSafelist;

/**
 * @public
 */
declare type StringRegExpArray = Array<RegExp | string>;

/**
 * @public
 */
export declare type UserDefinedOptions = Omit<UserDefinedOptions_2, "css" | "content" | "safelist" | "blocklist" | "sourceMap"> & {
    paths: string[] | PathFunction;
    moduleExtensions?: string[];
    verbose?: boolean;
    safelist: UserDefinedOptions_2["safelist"] | SafelistFunction;
    blocklist: UserDefinedOptions_2["blocklist"] | BlocklistFunction;
    only?: string[];
};

/**
 * Options used by PurgeCSS to remove unused CSS
 *
 * @public
 */
declare interface UserDefinedOptions_2 {
    /** {@inheritDoc Options.content} */
    content: Array<string | RawContent>;
    /** {@inheritDoc Options.css} */
    css: Array<string | RawCSS>;
    /** {@inheritDoc Options.defaultExtractor} */
    defaultExtractor?: ExtractorFunction;
    /** {@inheritDoc Options.extractors} */
    extractors?: Array<Extractors>;
    /** {@inheritDoc Options.fontFace} */
    fontFace?: boolean;
    /** {@inheritDoc Options.keyframes} */
    keyframes?: boolean;
    /** {@inheritDoc Options.output} */
    output?: string;
    /** {@inheritDoc Options.rejected} */
    rejected?: boolean;
    /** {@inheritDoc Options.rejectedCss} */
    rejectedCss?: boolean;
    /** {@inheritDoc Options.sourceMap } */
    sourceMap?: boolean | (postcss.SourceMapOptions & { to?: string });
    /** {@inheritDoc Options.stdin} */
    stdin?: boolean;
    /** {@inheritDoc Options.stdout} */
    stdout?: boolean;
    /** {@inheritDoc Options.variables} */
    variables?: boolean;
    /** {@inheritDoc Options.safelist} */
    safelist?: UserDefinedSafelist;
    /** {@inheritDoc Options.blocklist} */
    blocklist?: StringRegExpArray;
    /** {@inheritDoc Options.skippedContentGlobs} */
    skippedContentGlobs?: Array<string>;
    /** {@inheritDoc Options.dynamicAttributes} */
    dynamicAttributes?: string[];
}

/**
 * @public
 */
declare type UserDefinedSafelist = StringRegExpArray | ComplexSafelist;

export { }
