import { Deserializer, Serializer } from "./serialization/index.js";
import { Converter } from "./converter/index.js";
import { Renderer } from "./output/renderer.js";
import { type ProjectReflection } from "./models/index.js";
import { AbstractComponent, type OptionsReader } from "./utils/index.js";
import { Options } from "./utils/index.js";
import { type TypeDocOptions } from "./utils/options/declaration.js";
import { type GlobString, Logger } from "#utils";
import { type DocumentationEntryPoint, EntryPointStrategy } from "./utils/entry-point.js";
import { FileRegistry } from "./models/FileRegistry.js";
import { Outputs } from "./output/output.js";
import { Internationalization } from "./internationalization/internationalization.js";
export declare function createAppForTesting(): Application;
export interface ApplicationEvents {
    bootstrapEnd: [Application];
    reviveProject: [ProjectReflection];
    validateProject: [ProjectReflection];
}
/**
 * The default TypeDoc main application class.
 *
 * This class holds the two main components of TypeDoc, the {@link Converter} and
 * the {@link Renderer}. When running TypeDoc, first the {@link Converter} is invoked which
 * generates a {@link ProjectReflection} from the passed in source files. The
 * {@link ProjectReflection} is a hierarchical model representation of the TypeScript
 * project. Afterwards the model is passed to the {@link Renderer} which uses an instance
 * of {@link Theme} to generate the final documentation.
 *
 * Both the {@link Converter} and the {@link Renderer} emit a series of events while processing the project.
 * Subscribe to these Events to control the application flow or alter the output.
 *
 * @remarks
 *
 * Access to an Application instance can be retrieved with {@link Application.bootstrap} or
 * {@link Application.bootstrapWithPlugins}. It can not be constructed manually.
 *
 * @group None
 * @summary Root level class which contains most useful behavior.
 */
export declare class Application extends AbstractComponent<Application, ApplicationEvents> {
    private _logger;
    /**
     * The converter used to create the declaration reflections.
     */
    converter: Converter;
    outputs: Outputs;
    /**
     * The renderer used to generate the HTML documentation output.
     */
    renderer: Renderer;
    /**
     * The serializer used to generate JSON output.
     */
    serializer: Serializer;
    /**
     * The deserializer used to restore previously serialized JSON output.
     */
    deserializer: Deserializer;
    /**
     * The logger that should be used to output messages.
     */
    get logger(): Logger;
    set logger(l: Logger);
    /**
     * Internationalization module which supports translating according to
     * the `lang` option.
     */
    internationalization: Internationalization;
    options: Options;
    /**
     * Due for deprecation in 0.29, use the reference to this on {@link ProjectReflection},
     * this was the wrong place for this member to live.
     */
    files: FileRegistry;
    /** @internal */
    accessor lang: string;
    /** @internal */
    accessor skipErrorChecking: boolean;
    /** @internal */
    accessor entryPointStrategy: EntryPointStrategy;
    /** @internal */
    accessor entryPoints: GlobString[];
    /**
     * The version number of TypeDoc.
     */
    static readonly VERSION: string;
    /**
     * Emitted after plugins have been loaded and options have been read, but before they have been frozen.
     * The listener will be given an instance of {@link Application}.
     */
    static readonly EVENT_BOOTSTRAP_END: "bootstrapEnd";
    /**
     * Emitted after a project has been deserialized from JSON.
     * The listener will be given an instance of {@link ProjectReflection}.
     */
    static readonly EVENT_PROJECT_REVIVE: "reviveProject";
    /**
     * Emitted when validation is being run.
     * The listener will be given an instance of {@link ProjectReflection}.
     */
    static readonly EVENT_VALIDATE_PROJECT: "validateProject";
    /**
     * Create a new TypeDoc application instance.
     */
    private constructor();
    /**
     * Initialize TypeDoc, loading plugins if applicable.
     */
    static bootstrapWithPlugins(options?: TypeDocOptions, readers?: readonly OptionsReader[]): Promise<Application>;
    /**
     * Initialize TypeDoc without loading plugins.
     *
     * @example
     * Initialize the application with pretty-printing output disabled.
     * ```ts
     * const app = await Application.bootstrap({ pretty: false });
     * ```
     *
     * @param options Options to set during initialization
     * @param readers Option readers to use to discover options from config files.
     */
    static bootstrap(options?: TypeDocOptions, readers?: readonly OptionsReader[]): Promise<Application>;
    private _bootstrap;
    /** @internal */
    setOptions(options: TypeDocOptions, reportErrors?: boolean): boolean;
    /**
     * Return the path to the TypeScript compiler.
     */
    getTypeScriptPath(): string;
    getTypeScriptVersion(): string;
    getEntryPoints(): DocumentationEntryPoint[] | undefined;
    /**
     * Gets the entry points to be documented according to the current `entryPoints` and `entryPointStrategy` options.
     * May return undefined if entry points fail to be expanded.
     */
    getDefinedEntryPoints(): DocumentationEntryPoint[] | undefined;
    /**
     * Run the converter for the given set of files and return the generated reflections.
     *
     * @returns An instance of ProjectReflection on success, undefined otherwise.
     */
    convert(): Promise<ProjectReflection | undefined>;
    private watchers;
    private _watchFile?;
    private criticalFiles;
    private clearWatches;
    private watchConfigFile;
    /**
     * Register that the current build depends on a file, so that in watch mode
     * the build will be repeated.  Has no effect if a watch build is not
     * running, or if the file has already been registered.
     *
     * @param path The file to watch.  It does not need to exist, and you should
     * in fact register files you look for, but which do not exist, so that if
     * they are created the build will re-run.  (e.g. if you look through a list
     * of 5 possibilities and find the third, you should register the first 3.)
     *
     * @param shouldRestart Should the build be completely restarted?  (This is
     * normally only used for configuration files -- i.e. files whose contents
     * determine how conversion, rendering, or compiling will be done, as
     * opposed to files that are only read *during* the conversion or
     * rendering.)
     */
    watchFile(path: string, shouldRestart?: boolean): void;
    /**
     * Run a convert / watch process.
     *
     * @param success Callback to run after each convert, receiving the project
     * @returns True if the watch process should be restarted due to a
     * configuration change, false for an options error
     */
    convertAndWatch(success: (project: ProjectReflection) => Promise<void>): Promise<boolean>;
    validate(project: ProjectReflection): void;
    /**
     * Render outputs selected with options for the specified project
     */
    generateOutputs(project: ProjectReflection): Promise<void>;
    /**
     * Render HTML for the given project
     */
    generateDocs(project: ProjectReflection, out: string): Promise<void>;
    /**
     * Write the reflections to a json file.
     *
     * @param out The path and file name of the target file.
     * @returns Whether the JSON file could be written successfully.
     */
    generateJson(project: ProjectReflection, out: string): Promise<void>;
    /**
     * Print the version number.
     */
    toString(): string;
    private _convertPackages;
    private _merge;
}
