import * as onig from '@shikijs/engine-oniguruma';
import { readFile } from 'fs/promises';
import { dirname } from 'node:path';
import { fileURLToPath } from 'node:url';
import { ShikiError as ShikiError$1 } from '@shikijs/types';
import { INITIAL, EncodedTokenMetadata, FontStyle, Registry as Registry$1, Theme } from '@shikijs/vscode-textmate';

function resolveColorReplacements(theme, options) {
  const replacements = typeof theme === "string" ? {} : { ...theme.colorReplacements };
  const themeName = typeof theme === "string" ? theme : theme.name;
  for (const [key, value] of Object.entries(options?.colorReplacements || {})) {
    if (typeof value === "string")
      replacements[key] = value;
    else if (key === themeName)
      Object.assign(replacements, value);
  }
  return replacements;
}
function applyColorReplacements(color, replacements) {
  if (!color)
    return color;
  return replacements?.[color?.toLowerCase()] || color;
}

function toArray(x) {
  return Array.isArray(x) ? x : [x];
}
async function normalizeGetter(p) {
  return Promise.resolve(typeof p === "function" ? p() : p).then((r) => r.default || r);
}
function isPlainLang(lang) {
  return !lang || ["plaintext", "txt", "text", "plain"].includes(lang);
}
function isSpecialLang(lang) {
  return lang === "ansi" || isPlainLang(lang);
}
function isNoneTheme(theme) {
  return theme === "none";
}
function isSpecialTheme(theme) {
  return isNoneTheme(theme);
}

function splitLines(code, preserveEnding = false) {
  if (code.length === 0) {
    return [["", 0]];
  }
  const parts = code.split(/(\r?\n)/g);
  let index = 0;
  const lines = [];
  for (let i = 0; i < parts.length; i += 2) {
    const line = preserveEnding ? parts[i] + (parts[i + 1] || "") : parts[i];
    lines.push([line, index]);
    index += parts[i].length;
    index += parts[i + 1]?.length || 0;
  }
  return lines;
}

const _grammarStateMap = /* @__PURE__ */ new WeakMap();
function setLastGrammarStateToMap(keys, state) {
  _grammarStateMap.set(keys, state);
}
function getLastGrammarStateFromMap(keys) {
  return _grammarStateMap.get(keys);
}
class GrammarState {
  /**
   * Theme to Stack mapping
   */
  _stacks = {};
  lang;
  get themes() {
    return Object.keys(this._stacks);
  }
  get theme() {
    return this.themes[0];
  }
  get _stack() {
    return this._stacks[this.theme];
  }
  /**
   * Static method to create a initial grammar state.
   */
  static initial(lang, themes) {
    return new GrammarState(
      Object.fromEntries(toArray(themes).map((theme) => [theme, INITIAL])),
      lang
    );
  }
  constructor(...args) {
    if (args.length === 2) {
      const [stacksMap, lang] = args;
      this.lang = lang;
      this._stacks = stacksMap;
    } else {
      const [stack, lang, theme] = args;
      this.lang = lang;
      this._stacks = { [theme]: stack };
    }
  }
  /**
   * Get the internal stack object.
   * @internal
   */
  getInternalStack(theme = this.theme) {
    return this._stacks[theme];
  }
  getScopes(theme = this.theme) {
    return getScopes(this._stacks[theme]);
  }
  toJSON() {
    return {
      lang: this.lang,
      theme: this.theme,
      themes: this.themes,
      scopes: this.getScopes()
    };
  }
}
function getScopes(stack) {
  const scopes = [];
  const visited = /* @__PURE__ */ new Set();
  function pushScope(stack2) {
    if (visited.has(stack2))
      return;
    visited.add(stack2);
    const name = stack2?.nameScopesList?.scopeName;
    if (name)
      scopes.push(name);
    if (stack2.parent)
      pushScope(stack2.parent);
  }
  pushScope(stack);
  return scopes;
}
function getGrammarStack(state, theme) {
  if (!(state instanceof GrammarState))
    throw new ShikiError$1("Invalid grammar state");
  return state.getInternalStack(theme);
}

// src/colors.ts
var namedColors = [
  "black",
  "red",
  "green",
  "yellow",
  "blue",
  "magenta",
  "cyan",
  "white",
  "brightBlack",
  "brightRed",
  "brightGreen",
  "brightYellow",
  "brightBlue",
  "brightMagenta",
  "brightCyan",
  "brightWhite"
];

// src/decorations.ts
var decorations = {
  1: "bold",
  2: "dim",
  3: "italic",
  4: "underline",
  7: "reverse",
  8: "hidden",
  9: "strikethrough"
};

// src/parser.ts
function findSequence(value, position) {
  const nextEscape = value.indexOf("\x1B", position);
  if (nextEscape !== -1) {
    if (value[nextEscape + 1] === "[") {
      const nextClose = value.indexOf("m", nextEscape);
      if (nextClose !== -1) {
        return {
          sequence: value.substring(nextEscape + 2, nextClose).split(";"),
          startPosition: nextEscape,
          position: nextClose + 1
        };
      }
    }
  }
  return {
    position: value.length
  };
}
function parseColor(sequence) {
  const colorMode = sequence.shift();
  if (colorMode === "2") {
    const rgb = sequence.splice(0, 3).map((x) => Number.parseInt(x));
    if (rgb.length !== 3 || rgb.some((x) => Number.isNaN(x)))
      return;
    return {
      type: "rgb",
      rgb
    };
  } else if (colorMode === "5") {
    const index = sequence.shift();
    if (index) {
      return { type: "table", index: Number(index) };
    }
  }
}
function parseSequence(sequence) {
  const commands = [];
  while (sequence.length > 0) {
    const code = sequence.shift();
    if (!code)
      continue;
    const codeInt = Number.parseInt(code);
    if (Number.isNaN(codeInt))
      continue;
    if (codeInt === 0) {
      commands.push({ type: "resetAll" });
    } else if (codeInt <= 9) {
      const decoration = decorations[codeInt];
      if (decoration) {
        commands.push({
          type: "setDecoration",
          value: decorations[codeInt]
        });
      }
    } else if (codeInt <= 29) {
      const decoration = decorations[codeInt - 20];
      if (decoration) {
        commands.push({
          type: "resetDecoration",
          value: decoration
        });
        if (decoration === "dim") {
          commands.push({
            type: "resetDecoration",
            value: "bold"
          });
        }
      }
    } else if (codeInt <= 37) {
      commands.push({
        type: "setForegroundColor",
        value: { type: "named", name: namedColors[codeInt - 30] }
      });
    } else if (codeInt === 38) {
      const color = parseColor(sequence);
      if (color) {
        commands.push({
          type: "setForegroundColor",
          value: color
        });
      }
    } else if (codeInt === 39) {
      commands.push({
        type: "resetForegroundColor"
      });
    } else if (codeInt <= 47) {
      commands.push({
        type: "setBackgroundColor",
        value: { type: "named", name: namedColors[codeInt - 40] }
      });
    } else if (codeInt === 48) {
      const color = parseColor(sequence);
      if (color) {
        commands.push({
          type: "setBackgroundColor",
          value: color
        });
      }
    } else if (codeInt === 49) {
      commands.push({
        type: "resetBackgroundColor"
      });
    } else if (codeInt === 53) {
      commands.push({
        type: "setDecoration",
        value: "overline"
      });
    } else if (codeInt === 55) {
      commands.push({
        type: "resetDecoration",
        value: "overline"
      });
    } else if (codeInt >= 90 && codeInt <= 97) {
      commands.push({
        type: "setForegroundColor",
        value: { type: "named", name: namedColors[codeInt - 90 + 8] }
      });
    } else if (codeInt >= 100 && codeInt <= 107) {
      commands.push({
        type: "setBackgroundColor",
        value: { type: "named", name: namedColors[codeInt - 100 + 8] }
      });
    }
  }
  return commands;
}
function createAnsiSequenceParser() {
  let foreground = null;
  let background = null;
  let decorations2 = /* @__PURE__ */ new Set();
  return {
    parse(value) {
      const tokens = [];
      let position = 0;
      do {
        const findResult = findSequence(value, position);
        const text = findResult.sequence ? value.substring(position, findResult.startPosition) : value.substring(position);
        if (text.length > 0) {
          tokens.push({
            value: text,
            foreground,
            background,
            decorations: new Set(decorations2)
          });
        }
        if (findResult.sequence) {
          const commands = parseSequence(findResult.sequence);
          for (const styleToken of commands) {
            if (styleToken.type === "resetAll") {
              foreground = null;
              background = null;
              decorations2.clear();
            } else if (styleToken.type === "resetForegroundColor") {
              foreground = null;
            } else if (styleToken.type === "resetBackgroundColor") {
              background = null;
            } else if (styleToken.type === "resetDecoration") {
              decorations2.delete(styleToken.value);
            }
          }
          for (const styleToken of commands) {
            if (styleToken.type === "setForegroundColor") {
              foreground = styleToken.value;
            } else if (styleToken.type === "setBackgroundColor") {
              background = styleToken.value;
            } else if (styleToken.type === "setDecoration") {
              decorations2.add(styleToken.value);
            }
          }
        }
        position = findResult.position;
      } while (position < value.length);
      return tokens;
    }
  };
}

// src/palette.ts
var defaultNamedColorsMap = {
  black: "#000000",
  red: "#bb0000",
  green: "#00bb00",
  yellow: "#bbbb00",
  blue: "#0000bb",
  magenta: "#ff00ff",
  cyan: "#00bbbb",
  white: "#eeeeee",
  brightBlack: "#555555",
  brightRed: "#ff5555",
  brightGreen: "#00ff00",
  brightYellow: "#ffff55",
  brightBlue: "#5555ff",
  brightMagenta: "#ff55ff",
  brightCyan: "#55ffff",
  brightWhite: "#ffffff"
};
function createColorPalette(namedColorsMap = defaultNamedColorsMap) {
  function namedColor(name) {
    return namedColorsMap[name];
  }
  function rgbColor(rgb) {
    return `#${rgb.map((x) => Math.max(0, Math.min(x, 255)).toString(16).padStart(2, "0")).join("")}`;
  }
  let colorTable;
  function getColorTable() {
    if (colorTable) {
      return colorTable;
    }
    colorTable = [];
    for (let i = 0; i < namedColors.length; i++) {
      colorTable.push(namedColor(namedColors[i]));
    }
    let levels = [0, 95, 135, 175, 215, 255];
    for (let r = 0; r < 6; r++) {
      for (let g = 0; g < 6; g++) {
        for (let b = 0; b < 6; b++) {
          colorTable.push(rgbColor([levels[r], levels[g], levels[b]]));
        }
      }
    }
    let level = 8;
    for (let i = 0; i < 24; i++, level += 10) {
      colorTable.push(rgbColor([level, level, level]));
    }
    return colorTable;
  }
  function tableColor(index) {
    return getColorTable()[index];
  }
  function value(color) {
    switch (color.type) {
      case "named":
        return namedColor(color.name);
      case "rgb":
        return rgbColor(color.rgb);
      case "table":
        return tableColor(color.index);
    }
  }
  return {
    value
  };
}

const defaultAnsiColors = {
  black: "#000000",
  red: "#cd3131",
  green: "#0DBC79",
  yellow: "#E5E510",
  blue: "#2472C8",
  magenta: "#BC3FBC",
  cyan: "#11A8CD",
  white: "#E5E5E5",
  brightBlack: "#666666",
  brightRed: "#F14C4C",
  brightGreen: "#23D18B",
  brightYellow: "#F5F543",
  brightBlue: "#3B8EEA",
  brightMagenta: "#D670D6",
  brightCyan: "#29B8DB",
  brightWhite: "#FFFFFF"
};
function tokenizeAnsiWithTheme(theme, fileContents, options) {
  const colorReplacements = resolveColorReplacements(theme, options);
  const lines = splitLines(fileContents);
  const ansiPalette = Object.fromEntries(
    namedColors.map((name) => {
      const key = `terminal.ansi${name[0].toUpperCase()}${name.substring(1)}`;
      const themeColor = theme.colors?.[key];
      return [name, themeColor || defaultAnsiColors[name]];
    })
  );
  const colorPalette = createColorPalette(ansiPalette);
  const parser = createAnsiSequenceParser();
  return lines.map(
    (line) => parser.parse(line[0]).map((token) => {
      let color;
      let bgColor;
      if (token.decorations.has("reverse")) {
        color = token.background ? colorPalette.value(token.background) : theme.bg;
        bgColor = token.foreground ? colorPalette.value(token.foreground) : theme.fg;
      } else {
        color = token.foreground ? colorPalette.value(token.foreground) : theme.fg;
        bgColor = token.background ? colorPalette.value(token.background) : void 0;
      }
      color = applyColorReplacements(color, colorReplacements);
      bgColor = applyColorReplacements(bgColor, colorReplacements);
      if (token.decorations.has("dim"))
        color = dimColor(color);
      let fontStyle = FontStyle.None;
      if (token.decorations.has("bold"))
        fontStyle |= FontStyle.Bold;
      if (token.decorations.has("italic"))
        fontStyle |= FontStyle.Italic;
      if (token.decorations.has("underline"))
        fontStyle |= FontStyle.Underline;
      if (token.decorations.has("strikethrough"))
        fontStyle |= FontStyle.Strikethrough;
      return {
        content: token.value,
        offset: line[1],
        // TODO: more accurate offset? might need to fork ansi-sequence-parser
        color,
        bgColor,
        fontStyle
      };
    })
  );
}
function dimColor(color) {
  const hexMatch = color.match(/#([0-9a-f]{3,8})/i);
  if (hexMatch) {
    const hex = hexMatch[1];
    if (hex.length === 8) {
      const alpha = Math.round(Number.parseInt(hex.slice(6, 8), 16) / 2).toString(16).padStart(2, "0");
      return `#${hex.slice(0, 6)}${alpha}`;
    } else if (hex.length === 6) {
      return `#${hex}80`;
    } else if (hex.length === 4) {
      const r = hex[0];
      const g = hex[1];
      const b = hex[2];
      const a = hex[3];
      const alpha = Math.round(Number.parseInt(`${a}${a}`, 16) / 2).toString(16).padStart(2, "0");
      return `#${r}${r}${g}${g}${b}${b}${alpha}`;
    } else if (hex.length === 3) {
      const r = hex[0];
      const g = hex[1];
      const b = hex[2];
      return `#${r}${r}${g}${g}${b}${b}80`;
    }
  }
  const cssVarMatch = color.match(/var\((--[\w-]+-ansi-[\w-]+)\)/);
  if (cssVarMatch)
    return `var(${cssVarMatch[1]}-dim)`;
  return color;
}

function codeToTokensBase(internal, code, options = {}) {
  const {
    theme: themeName = internal.getLoadedThemes()[0]
  } = options;
  const lang = internal.resolveLangAlias(options.lang || "text");
  if (isPlainLang(lang) || isNoneTheme(themeName))
    return splitLines(code).map((line) => [{ content: line[0], offset: line[1] }]);
  const { theme, colorMap } = internal.setTheme(themeName);
  if (lang === "ansi")
    return tokenizeAnsiWithTheme(theme, code, options);
  const _grammar = internal.getLanguage(options.lang || "text");
  if (options.grammarState) {
    if (options.grammarState.lang !== _grammar.name) {
      throw new ShikiError$1(`Grammar state language "${options.grammarState.lang}" does not match highlight language "${_grammar.name}"`);
    }
    if (!options.grammarState.themes.includes(theme.name)) {
      throw new ShikiError$1(`Grammar state themes "${options.grammarState.themes}" do not contain highlight theme "${theme.name}"`);
    }
  }
  return tokenizeWithTheme(code, _grammar, theme, colorMap, options);
}
function tokenizeWithTheme(code, grammar, theme, colorMap, options) {
  const result = _tokenizeWithTheme(code, grammar, theme, colorMap, options);
  const grammarState = new GrammarState(
    result.stateStack,
    grammar.name,
    theme.name
  );
  setLastGrammarStateToMap(result.tokens, grammarState);
  return result.tokens;
}
function _tokenizeWithTheme(code, grammar, theme, colorMap, options) {
  const colorReplacements = resolveColorReplacements(theme, options);
  const {
    tokenizeMaxLineLength = 0,
    tokenizeTimeLimit = 500
  } = options;
  const lines = splitLines(code);
  let stateStack = options.grammarState ? getGrammarStack(options.grammarState, theme.name) ?? INITIAL : options.grammarContextCode != null ? _tokenizeWithTheme(
    options.grammarContextCode,
    grammar,
    theme,
    colorMap,
    {
      ...options,
      grammarState: void 0,
      grammarContextCode: void 0
    }
  ).stateStack : INITIAL;
  let actual = [];
  const final = [];
  for (let i = 0, len = lines.length; i < len; i++) {
    const [line, lineOffset] = lines[i];
    if (line === "") {
      actual = [];
      final.push([]);
      continue;
    }
    if (tokenizeMaxLineLength > 0 && line.length >= tokenizeMaxLineLength) {
      actual = [];
      final.push([{
        content: line,
        offset: lineOffset,
        color: "",
        fontStyle: 0
      }]);
      continue;
    }
    let resultWithScopes;
    let tokensWithScopes;
    let tokensWithScopesIndex;
    if (options.includeExplanation) {
      resultWithScopes = grammar.tokenizeLine(line, stateStack, tokenizeTimeLimit);
      tokensWithScopes = resultWithScopes.tokens;
      tokensWithScopesIndex = 0;
    }
    const result = grammar.tokenizeLine2(line, stateStack, tokenizeTimeLimit);
    const tokensLength = result.tokens.length / 2;
    for (let j = 0; j < tokensLength; j++) {
      const startIndex = result.tokens[2 * j];
      const nextStartIndex = j + 1 < tokensLength ? result.tokens[2 * j + 2] : line.length;
      if (startIndex === nextStartIndex)
        continue;
      const metadata = result.tokens[2 * j + 1];
      const color = applyColorReplacements(
        colorMap[EncodedTokenMetadata.getForeground(metadata)],
        colorReplacements
      );
      const fontStyle = EncodedTokenMetadata.getFontStyle(metadata);
      const token = {
        content: line.substring(startIndex, nextStartIndex),
        offset: lineOffset + startIndex,
        color,
        fontStyle
      };
      if (options.includeExplanation) {
        const themeSettingsSelectors = [];
        if (options.includeExplanation !== "scopeName") {
          for (const setting of theme.settings) {
            let selectors;
            switch (typeof setting.scope) {
              case "string":
                selectors = setting.scope.split(/,/).map((scope) => scope.trim());
                break;
              case "object":
                selectors = setting.scope;
                break;
              default:
                continue;
            }
            themeSettingsSelectors.push({
              settings: setting,
              selectors: selectors.map((selector) => selector.split(/ /))
            });
          }
        }
        token.explanation = [];
        let offset = 0;
        while (startIndex + offset < nextStartIndex) {
          const tokenWithScopes = tokensWithScopes[tokensWithScopesIndex];
          const tokenWithScopesText = line.substring(
            tokenWithScopes.startIndex,
            tokenWithScopes.endIndex
          );
          offset += tokenWithScopesText.length;
          token.explanation.push({
            content: tokenWithScopesText,
            scopes: options.includeExplanation === "scopeName" ? explainThemeScopesNameOnly(
              tokenWithScopes.scopes
            ) : explainThemeScopesFull(
              themeSettingsSelectors,
              tokenWithScopes.scopes
            )
          });
          tokensWithScopesIndex += 1;
        }
      }
      actual.push(token);
    }
    final.push(actual);
    actual = [];
    stateStack = result.ruleStack;
  }
  return {
    tokens: final,
    stateStack
  };
}
function explainThemeScopesNameOnly(scopes) {
  return scopes.map((scope) => ({ scopeName: scope }));
}
function explainThemeScopesFull(themeSelectors, scopes) {
  const result = [];
  for (let i = 0, len = scopes.length; i < len; i++) {
    const scope = scopes[i];
    result[i] = {
      scopeName: scope,
      themeMatches: explainThemeScope(themeSelectors, scope, scopes.slice(0, i))
    };
  }
  return result;
}
function matchesOne(selector, scope) {
  return selector === scope || scope.substring(0, selector.length) === selector && scope[selector.length] === ".";
}
function matches(selectors, scope, parentScopes) {
  if (!matchesOne(selectors[selectors.length - 1], scope))
    return false;
  let selectorParentIndex = selectors.length - 2;
  let parentIndex = parentScopes.length - 1;
  while (selectorParentIndex >= 0 && parentIndex >= 0) {
    if (matchesOne(selectors[selectorParentIndex], parentScopes[parentIndex]))
      selectorParentIndex -= 1;
    parentIndex -= 1;
  }
  if (selectorParentIndex === -1)
    return true;
  return false;
}
function explainThemeScope(themeSettingsSelectors, scope, parentScopes) {
  const result = [];
  for (const { selectors, settings } of themeSettingsSelectors) {
    for (const selectorPieces of selectors) {
      if (matches(selectorPieces, scope, parentScopes)) {
        result.push(settings);
        break;
      }
    }
  }
  return result;
}

function codeToTokensWithThemes(internal, code, options) {
  const themes = Object.entries(options.themes).filter((i) => i[1]).map((i) => ({ color: i[0], theme: i[1] }));
  const themedTokens = themes.map((t) => {
    const tokens2 = codeToTokensBase(internal, code, {
      ...options,
      theme: t.theme
    });
    const state = getLastGrammarStateFromMap(tokens2);
    const theme = typeof t.theme === "string" ? t.theme : t.theme.name;
    return {
      tokens: tokens2,
      state,
      theme
    };
  });
  const tokens = syncThemesTokenization(
    ...themedTokens.map((i) => i.tokens)
  );
  const mergedTokens = tokens[0].map(
    (line, lineIdx) => line.map((_token, tokenIdx) => {
      const mergedToken = {
        content: _token.content,
        variants: {},
        offset: _token.offset
      };
      if ("includeExplanation" in options && options.includeExplanation) {
        mergedToken.explanation = _token.explanation;
      }
      tokens.forEach((t, themeIdx) => {
        const {
          content: _,
          explanation: __,
          offset: ___,
          ...styles
        } = t[lineIdx][tokenIdx];
        mergedToken.variants[themes[themeIdx].color] = styles;
      });
      return mergedToken;
    })
  );
  const mergedGrammarState = themedTokens[0].state ? new GrammarState(
    Object.fromEntries(themedTokens.map((s) => [s.theme, s.state?.getInternalStack(s.theme)])),
    themedTokens[0].state.lang
  ) : void 0;
  if (mergedGrammarState)
    setLastGrammarStateToMap(mergedTokens, mergedGrammarState);
  return mergedTokens;
}
function syncThemesTokenization(...themes) {
  const outThemes = themes.map(() => []);
  const count = themes.length;
  for (let i = 0; i < themes[0].length; i++) {
    const lines = themes.map((t) => t[i]);
    const outLines = outThemes.map(() => []);
    outThemes.forEach((t, i2) => t.push(outLines[i2]));
    const indexes = lines.map(() => 0);
    const current = lines.map((l) => l[0]);
    while (current.every((t) => t)) {
      const minLength = Math.min(...current.map((t) => t.content.length));
      for (let n = 0; n < count; n++) {
        const token = current[n];
        if (token.content.length === minLength) {
          outLines[n].push(token);
          indexes[n] += 1;
          current[n] = lines[n][indexes[n]];
        } else {
          outLines[n].push({
            ...token,
            content: token.content.slice(0, minLength)
          });
          current[n] = {
            ...token,
            content: token.content.slice(minLength),
            offset: token.offset + minLength
          };
        }
      }
    }
  }
  return outThemes;
}

const VSCODE_FALLBACK_EDITOR_FG = { light: "#333333", dark: "#bbbbbb" };
const VSCODE_FALLBACK_EDITOR_BG = { light: "#fffffe", dark: "#1e1e1e" };
const RESOLVED_KEY = "__shiki_resolved";
function normalizeTheme(rawTheme) {
  if (rawTheme?.[RESOLVED_KEY])
    return rawTheme;
  const theme = {
    ...rawTheme
  };
  if (theme.tokenColors && !theme.settings) {
    theme.settings = theme.tokenColors;
    delete theme.tokenColors;
  }
  theme.type ||= "dark";
  theme.colorReplacements = { ...theme.colorReplacements };
  theme.settings ||= [];
  let { bg, fg } = theme;
  if (!bg || !fg) {
    const globalSetting = theme.settings ? theme.settings.find((s) => !s.name && !s.scope) : void 0;
    if (globalSetting?.settings?.foreground)
      fg = globalSetting.settings.foreground;
    if (globalSetting?.settings?.background)
      bg = globalSetting.settings.background;
    if (!fg && theme?.colors?.["editor.foreground"])
      fg = theme.colors["editor.foreground"];
    if (!bg && theme?.colors?.["editor.background"])
      bg = theme.colors["editor.background"];
    if (!fg)
      fg = theme.type === "light" ? VSCODE_FALLBACK_EDITOR_FG.light : VSCODE_FALLBACK_EDITOR_FG.dark;
    if (!bg)
      bg = theme.type === "light" ? VSCODE_FALLBACK_EDITOR_BG.light : VSCODE_FALLBACK_EDITOR_BG.dark;
    theme.fg = fg;
    theme.bg = bg;
  }
  if (!(theme.settings[0] && theme.settings[0].settings && !theme.settings[0].scope)) {
    theme.settings.unshift({
      settings: {
        foreground: theme.fg,
        background: theme.bg
      }
    });
  }
  let replacementCount = 0;
  const replacementMap = /* @__PURE__ */ new Map();
  function getReplacementColor(value) {
    if (replacementMap.has(value))
      return replacementMap.get(value);
    replacementCount += 1;
    const hex = `#${replacementCount.toString(16).padStart(8, "0").toLowerCase()}`;
    if (theme.colorReplacements?.[`#${hex}`])
      return getReplacementColor(value);
    replacementMap.set(value, hex);
    return hex;
  }
  theme.settings = theme.settings.map((setting) => {
    const replaceFg = setting.settings?.foreground && !setting.settings.foreground.startsWith("#");
    const replaceBg = setting.settings?.background && !setting.settings.background.startsWith("#");
    if (!replaceFg && !replaceBg)
      return setting;
    const clone = {
      ...setting,
      settings: {
        ...setting.settings
      }
    };
    if (replaceFg) {
      const replacement = getReplacementColor(setting.settings.foreground);
      theme.colorReplacements[replacement] = setting.settings.foreground;
      clone.settings.foreground = replacement;
    }
    if (replaceBg) {
      const replacement = getReplacementColor(setting.settings.background);
      theme.colorReplacements[replacement] = setting.settings.background;
      clone.settings.background = replacement;
    }
    return clone;
  });
  for (const key of Object.keys(theme.colors || {})) {
    if (key === "editor.foreground" || key === "editor.background" || key.startsWith("terminal.ansi")) {
      if (!theme.colors[key]?.startsWith("#")) {
        const replacement = getReplacementColor(theme.colors[key]);
        theme.colorReplacements[replacement] = theme.colors[key];
        theme.colors[key] = replacement;
      }
    }
  }
  Object.defineProperty(theme, RESOLVED_KEY, {
    enumerable: false,
    writable: false,
    value: true
  });
  return theme;
}

async function resolveLangs(langs) {
  return Array.from(new Set((await Promise.all(
    langs.filter((l) => !isSpecialLang(l)).map(async (lang) => await normalizeGetter(lang).then((r) => Array.isArray(r) ? r : [r]))
  )).flat()));
}
async function resolveThemes(themes) {
  const resolved = await Promise.all(
    themes.map(
      async (theme) => isSpecialTheme(theme) ? null : normalizeTheme(await normalizeGetter(theme))
    )
  );
  return resolved.filter((i) => !!i);
}

let _emitDeprecation = 3;
function warnDeprecated(message, version = 3) {
  if (version > _emitDeprecation)
    return;
  {
    console.trace(`[SHIKI DEPRECATE]: ${message}`);
  }
}

class ShikiError extends Error {
  constructor(message) {
    super(message);
    this.name = "ShikiError";
  }
}

function resolveLangAlias(name, alias) {
  if (!alias)
    return name;
  if (alias[name]) {
    const resolved = /* @__PURE__ */ new Set([name]);
    while (alias[name]) {
      name = alias[name];
      if (resolved.has(name))
        throw new ShikiError(`Circular alias \`${Array.from(resolved).join(" -> ")} -> ${name}\``);
      resolved.add(name);
    }
  }
  return name;
}

class Registry extends Registry$1 {
  constructor(_resolver, _themes, _langs, _alias = {}) {
    super(_resolver);
    this._resolver = _resolver;
    this._themes = _themes;
    this._langs = _langs;
    this._alias = _alias;
    this._themes.map((t) => this.loadTheme(t));
    this.loadLanguages(this._langs);
  }
  _resolvedThemes = /* @__PURE__ */ new Map();
  _resolvedGrammars = /* @__PURE__ */ new Map();
  _langMap = /* @__PURE__ */ new Map();
  _langGraph = /* @__PURE__ */ new Map();
  _textmateThemeCache = /* @__PURE__ */ new WeakMap();
  _loadedThemesCache = null;
  _loadedLanguagesCache = null;
  getTheme(theme) {
    if (typeof theme === "string")
      return this._resolvedThemes.get(theme);
    else
      return this.loadTheme(theme);
  }
  loadTheme(theme) {
    const _theme = normalizeTheme(theme);
    if (_theme.name) {
      this._resolvedThemes.set(_theme.name, _theme);
      this._loadedThemesCache = null;
    }
    return _theme;
  }
  getLoadedThemes() {
    if (!this._loadedThemesCache)
      this._loadedThemesCache = [...this._resolvedThemes.keys()];
    return this._loadedThemesCache;
  }
  // Override and re-implement this method to cache the textmate themes as `TextMateTheme.createFromRawTheme`
  // is expensive. Themes can switch often especially for dual-theme support.
  //
  // The parent class also accepts `colorMap` as the second parameter, but since we don't use that,
  // we omit here so it's easier to cache the themes.
  setTheme(theme) {
    let textmateTheme = this._textmateThemeCache.get(theme);
    if (!textmateTheme) {
      textmateTheme = Theme.createFromRawTheme(theme);
      this._textmateThemeCache.set(theme, textmateTheme);
    }
    this._syncRegistry.setTheme(textmateTheme);
  }
  getGrammar(name) {
    name = resolveLangAlias(name, this._alias);
    return this._resolvedGrammars.get(name);
  }
  loadLanguage(lang) {
    if (this.getGrammar(lang.name))
      return;
    const embeddedLazilyBy = new Set(
      [...this._langMap.values()].filter((i) => i.embeddedLangsLazy?.includes(lang.name))
    );
    this._resolver.addLanguage(lang);
    const grammarConfig = {
      balancedBracketSelectors: lang.balancedBracketSelectors || ["*"],
      unbalancedBracketSelectors: lang.unbalancedBracketSelectors || []
    };
    this._syncRegistry._rawGrammars.set(lang.scopeName, lang);
    const g = this.loadGrammarWithConfiguration(lang.scopeName, 1, grammarConfig);
    g.name = lang.name;
    this._resolvedGrammars.set(lang.name, g);
    if (lang.aliases) {
      lang.aliases.forEach((alias) => {
        this._alias[alias] = lang.name;
      });
    }
    this._loadedLanguagesCache = null;
    if (embeddedLazilyBy.size) {
      for (const e of embeddedLazilyBy) {
        this._resolvedGrammars.delete(e.name);
        this._loadedLanguagesCache = null;
        this._syncRegistry?._injectionGrammars?.delete(e.scopeName);
        this._syncRegistry?._grammars?.delete(e.scopeName);
        this.loadLanguage(this._langMap.get(e.name));
      }
    }
  }
  dispose() {
    super.dispose();
    this._resolvedThemes.clear();
    this._resolvedGrammars.clear();
    this._langMap.clear();
    this._langGraph.clear();
    this._loadedThemesCache = null;
  }
  loadLanguages(langs) {
    for (const lang of langs)
      this.resolveEmbeddedLanguages(lang);
    const langsGraphArray = Array.from(this._langGraph.entries());
    const missingLangs = langsGraphArray.filter(([_, lang]) => !lang);
    if (missingLangs.length) {
      const dependents = langsGraphArray.filter(([_, lang]) => {
        if (!lang)
          return false;
        const embedded = lang.embeddedLanguages || lang.embeddedLangs;
        return embedded?.some((l) => missingLangs.map(([name]) => name).includes(l));
      }).filter((lang) => !missingLangs.includes(lang));
      throw new ShikiError(`Missing languages ${missingLangs.map(([name]) => `\`${name}\``).join(", ")}, required by ${dependents.map(([name]) => `\`${name}\``).join(", ")}`);
    }
    for (const [_, lang] of langsGraphArray)
      this._resolver.addLanguage(lang);
    for (const [_, lang] of langsGraphArray)
      this.loadLanguage(lang);
  }
  getLoadedLanguages() {
    if (!this._loadedLanguagesCache) {
      this._loadedLanguagesCache = [
        .../* @__PURE__ */ new Set([...this._resolvedGrammars.keys(), ...Object.keys(this._alias)])
      ];
    }
    return this._loadedLanguagesCache;
  }
  resolveEmbeddedLanguages(lang) {
    this._langMap.set(lang.name, lang);
    this._langGraph.set(lang.name, lang);
    const embedded = lang.embeddedLanguages ?? lang.embeddedLangs;
    if (embedded) {
      for (const embeddedLang of embedded)
        this._langGraph.set(embeddedLang, this._langMap.get(embeddedLang));
    }
  }
}

class Resolver {
  _langs = /* @__PURE__ */ new Map();
  _scopeToLang = /* @__PURE__ */ new Map();
  _injections = /* @__PURE__ */ new Map();
  _onigLib;
  constructor(engine, langs) {
    this._onigLib = {
      createOnigScanner: (patterns) => engine.createScanner(patterns),
      createOnigString: (s) => engine.createString(s)
    };
    langs.forEach((i) => this.addLanguage(i));
  }
  get onigLib() {
    return this._onigLib;
  }
  getLangRegistration(langIdOrAlias) {
    return this._langs.get(langIdOrAlias);
  }
  loadGrammar(scopeName) {
    return this._scopeToLang.get(scopeName);
  }
  addLanguage(l) {
    this._langs.set(l.name, l);
    if (l.aliases) {
      l.aliases.forEach((a) => {
        this._langs.set(a, l);
      });
    }
    this._scopeToLang.set(l.scopeName, l);
    if (l.injectTo) {
      l.injectTo.forEach((i) => {
        if (!this._injections.get(i))
          this._injections.set(i, []);
        this._injections.get(i).push(l.scopeName);
      });
    }
  }
  getInjections(scopeName) {
    const scopeParts = scopeName.split(".");
    let injections = [];
    for (let i = 1; i <= scopeParts.length; i++) {
      const subScopeName = scopeParts.slice(0, i).join(".");
      injections = [...injections, ...this._injections.get(subScopeName) || []];
    }
    return injections;
  }
}

let instancesCount = 0;
function createShikiInternalSync(options) {
  instancesCount += 1;
  if (options.warnings !== false && instancesCount >= 10 && instancesCount % 10 === 0)
    console.warn(`[Shiki] ${instancesCount} instances have been created. Shiki is supposed to be used as a singleton, consider refactoring your code to cache your highlighter instance; Or call \`highlighter.dispose()\` to release unused instances.`);
  let isDisposed = false;
  if (!options.engine)
    throw new ShikiError("`engine` option is required for synchronous mode");
  const langs = (options.langs || []).flat(1);
  const themes = (options.themes || []).flat(1).map(normalizeTheme);
  const resolver = new Resolver(options.engine, langs);
  const _registry = new Registry(resolver, themes, langs, options.langAlias);
  let _lastTheme;
  function resolveLangAlias$1(name) {
    return resolveLangAlias(name, options.langAlias);
  }
  function getLanguage(name) {
    ensureNotDisposed();
    const _lang = _registry.getGrammar(typeof name === "string" ? name : name.name);
    if (!_lang)
      throw new ShikiError(`Language \`${name}\` not found, you may need to load it first`);
    return _lang;
  }
  function getTheme(name) {
    if (name === "none")
      return { bg: "", fg: "", name: "none", settings: [], type: "dark" };
    ensureNotDisposed();
    const _theme = _registry.getTheme(name);
    if (!_theme)
      throw new ShikiError(`Theme \`${name}\` not found, you may need to load it first`);
    return _theme;
  }
  function setTheme(name) {
    ensureNotDisposed();
    const theme = getTheme(name);
    if (_lastTheme !== name) {
      _registry.setTheme(theme);
      _lastTheme = name;
    }
    const colorMap = _registry.getColorMap();
    return {
      theme,
      colorMap
    };
  }
  function getLoadedThemes() {
    ensureNotDisposed();
    return _registry.getLoadedThemes();
  }
  function getLoadedLanguages() {
    ensureNotDisposed();
    return _registry.getLoadedLanguages();
  }
  function loadLanguageSync(...langs2) {
    ensureNotDisposed();
    _registry.loadLanguages(langs2.flat(1));
  }
  async function loadLanguage(...langs2) {
    return loadLanguageSync(await resolveLangs(langs2));
  }
  function loadThemeSync(...themes2) {
    ensureNotDisposed();
    for (const theme of themes2.flat(1)) {
      _registry.loadTheme(theme);
    }
  }
  async function loadTheme(...themes2) {
    ensureNotDisposed();
    return loadThemeSync(await resolveThemes(themes2));
  }
  function ensureNotDisposed() {
    if (isDisposed)
      throw new ShikiError("Shiki instance has been disposed");
  }
  function dispose() {
    if (isDisposed)
      return;
    isDisposed = true;
    _registry.dispose();
    instancesCount -= 1;
  }
  return {
    setTheme,
    getTheme,
    getLanguage,
    getLoadedThemes,
    getLoadedLanguages,
    resolveLangAlias: resolveLangAlias$1,
    loadLanguage,
    loadLanguageSync,
    loadTheme,
    loadThemeSync,
    dispose,
    [Symbol.dispose]: dispose
  };
}

async function createShikiInternal(options) {
  if (!options.engine) {
    warnDeprecated("`engine` option is required. Use `createOnigurumaEngine` or `createJavaScriptRegexEngine` to create an engine.");
  }
  const [
    themes,
    langs,
    engine
  ] = await Promise.all([
    resolveThemes(options.themes || []),
    resolveLangs(options.langs || []),
    options.engine
  ]);
  return createShikiInternalSync({
    ...options,
    themes,
    langs,
    engine
  });
}

const bundledLanguagesInfo = [
  {
    "id": "abap",
    "name": "ABAP",
    "import": (() => import('@shikijs/langs/abap'))
  },
  {
    "id": "actionscript-3",
    "name": "ActionScript",
    "import": (() => import('@shikijs/langs/actionscript-3'))
  },
  {
    "id": "ada",
    "name": "Ada",
    "import": (() => import('@shikijs/langs/ada'))
  },
  {
    "id": "angular-html",
    "name": "Angular HTML",
    "import": (() => import('@shikijs/langs/angular-html'))
  },
  {
    "id": "angular-ts",
    "name": "Angular TypeScript",
    "import": (() => import('@shikijs/langs/angular-ts'))
  },
  {
    "id": "apache",
    "name": "Apache Conf",
    "import": (() => import('@shikijs/langs/apache'))
  },
  {
    "id": "apex",
    "name": "Apex",
    "import": (() => import('@shikijs/langs/apex'))
  },
  {
    "id": "apl",
    "name": "APL",
    "import": (() => import('@shikijs/langs/apl'))
  },
  {
    "id": "applescript",
    "name": "AppleScript",
    "import": (() => import('@shikijs/langs/applescript'))
  },
  {
    "id": "ara",
    "name": "Ara",
    "import": (() => import('@shikijs/langs/ara'))
  },
  {
    "id": "asciidoc",
    "name": "AsciiDoc",
    "aliases": [
      "adoc"
    ],
    "import": (() => import('@shikijs/langs/asciidoc'))
  },
  {
    "id": "asm",
    "name": "Assembly",
    "import": (() => import('@shikijs/langs/asm'))
  },
  {
    "id": "astro",
    "name": "Astro",
    "import": (() => import('@shikijs/langs/astro'))
  },
  {
    "id": "awk",
    "name": "AWK",
    "import": (() => import('@shikijs/langs/awk'))
  },
  {
    "id": "ballerina",
    "name": "Ballerina",
    "import": (() => import('@shikijs/langs/ballerina'))
  },
  {
    "id": "bat",
    "name": "Batch File",
    "aliases": [
      "batch"
    ],
    "import": (() => import('@shikijs/langs/bat'))
  },
  {
    "id": "beancount",
    "name": "Beancount",
    "import": (() => import('@shikijs/langs/beancount'))
  },
  {
    "id": "berry",
    "name": "Berry",
    "aliases": [
      "be"
    ],
    "import": (() => import('@shikijs/langs/berry'))
  },
  {
    "id": "bibtex",
    "name": "BibTeX",
    "import": (() => import('@shikijs/langs/bibtex'))
  },
  {
    "id": "bicep",
    "name": "Bicep",
    "import": (() => import('@shikijs/langs/bicep'))
  },
  {
    "id": "blade",
    "name": "Blade",
    "import": (() => import('@shikijs/langs/blade'))
  },
  {
    "id": "bsl",
    "name": "1C (Enterprise)",
    "aliases": [
      "1c"
    ],
    "import": (() => import('@shikijs/langs/bsl'))
  },
  {
    "id": "c",
    "name": "C",
    "import": (() => import('@shikijs/langs/c'))
  },
  {
    "id": "cadence",
    "name": "Cadence",
    "aliases": [
      "cdc"
    ],
    "import": (() => import('@shikijs/langs/cadence'))
  },
  {
    "id": "cairo",
    "name": "Cairo",
    "import": (() => import('@shikijs/langs/cairo'))
  },
  {
    "id": "clarity",
    "name": "Clarity",
    "import": (() => import('@shikijs/langs/clarity'))
  },
  {
    "id": "clojure",
    "name": "Clojure",
    "aliases": [
      "clj"
    ],
    "import": (() => import('@shikijs/langs/clojure'))
  },
  {
    "id": "cmake",
    "name": "CMake",
    "import": (() => import('@shikijs/langs/cmake'))
  },
  {
    "id": "cobol",
    "name": "COBOL",
    "import": (() => import('@shikijs/langs/cobol'))
  },
  {
    "id": "codeowners",
    "name": "CODEOWNERS",
    "import": (() => import('@shikijs/langs/codeowners'))
  },
  {
    "id": "codeql",
    "name": "CodeQL",
    "aliases": [
      "ql"
    ],
    "import": (() => import('@shikijs/langs/codeql'))
  },
  {
    "id": "coffee",
    "name": "CoffeeScript",
    "aliases": [
      "coffeescript"
    ],
    "import": (() => import('@shikijs/langs/coffee'))
  },
  {
    "id": "common-lisp",
    "name": "Common Lisp",
    "aliases": [
      "lisp"
    ],
    "import": (() => import('@shikijs/langs/common-lisp'))
  },
  {
    "id": "coq",
    "name": "Coq",
    "import": (() => import('@shikijs/langs/coq'))
  },
  {
    "id": "cpp",
    "name": "C++",
    "aliases": [
      "c++"
    ],
    "import": (() => import('@shikijs/langs/cpp'))
  },
  {
    "id": "crystal",
    "name": "Crystal",
    "import": (() => import('@shikijs/langs/crystal'))
  },
  {
    "id": "csharp",
    "name": "C#",
    "aliases": [
      "c#",
      "cs"
    ],
    "import": (() => import('@shikijs/langs/csharp'))
  },
  {
    "id": "css",
    "name": "CSS",
    "import": (() => import('@shikijs/langs/css'))
  },
  {
    "id": "csv",
    "name": "CSV",
    "import": (() => import('@shikijs/langs/csv'))
  },
  {
    "id": "cue",
    "name": "CUE",
    "import": (() => import('@shikijs/langs/cue'))
  },
  {
    "id": "cypher",
    "name": "Cypher",
    "aliases": [
      "cql"
    ],
    "import": (() => import('@shikijs/langs/cypher'))
  },
  {
    "id": "d",
    "name": "D",
    "import": (() => import('@shikijs/langs/d'))
  },
  {
    "id": "dart",
    "name": "Dart",
    "import": (() => import('@shikijs/langs/dart'))
  },
  {
    "id": "dax",
    "name": "DAX",
    "import": (() => import('@shikijs/langs/dax'))
  },
  {
    "id": "desktop",
    "name": "Desktop",
    "import": (() => import('@shikijs/langs/desktop'))
  },
  {
    "id": "diff",
    "name": "Diff",
    "import": (() => import('@shikijs/langs/diff'))
  },
  {
    "id": "docker",
    "name": "Dockerfile",
    "aliases": [
      "dockerfile"
    ],
    "import": (() => import('@shikijs/langs/docker'))
  },
  {
    "id": "dotenv",
    "name": "dotEnv",
    "import": (() => import('@shikijs/langs/dotenv'))
  },
  {
    "id": "dream-maker",
    "name": "Dream Maker",
    "import": (() => import('@shikijs/langs/dream-maker'))
  },
  {
    "id": "edge",
    "name": "Edge",
    "import": (() => import('@shikijs/langs/edge'))
  },
  {
    "id": "elixir",
    "name": "Elixir",
    "import": (() => import('@shikijs/langs/elixir'))
  },
  {
    "id": "elm",
    "name": "Elm",
    "import": (() => import('@shikijs/langs/elm'))
  },
  {
    "id": "emacs-lisp",
    "name": "Emacs Lisp",
    "aliases": [
      "elisp"
    ],
    "import": (() => import('@shikijs/langs/emacs-lisp'))
  },
  {
    "id": "erb",
    "name": "ERB",
    "import": (() => import('@shikijs/langs/erb'))
  },
  {
    "id": "erlang",
    "name": "Erlang",
    "aliases": [
      "erl"
    ],
    "import": (() => import('@shikijs/langs/erlang'))
  },
  {
    "id": "fennel",
    "name": "Fennel",
    "import": (() => import('@shikijs/langs/fennel'))
  },
  {
    "id": "fish",
    "name": "Fish",
    "import": (() => import('@shikijs/langs/fish'))
  },
  {
    "id": "fluent",
    "name": "Fluent",
    "aliases": [
      "ftl"
    ],
    "import": (() => import('@shikijs/langs/fluent'))
  },
  {
    "id": "fortran-fixed-form",
    "name": "Fortran (Fixed Form)",
    "aliases": [
      "f",
      "for",
      "f77"
    ],
    "import": (() => import('@shikijs/langs/fortran-fixed-form'))
  },
  {
    "id": "fortran-free-form",
    "name": "Fortran (Free Form)",
    "aliases": [
      "f90",
      "f95",
      "f03",
      "f08",
      "f18"
    ],
    "import": (() => import('@shikijs/langs/fortran-free-form'))
  },
  {
    "id": "fsharp",
    "name": "F#",
    "aliases": [
      "f#",
      "fs"
    ],
    "import": (() => import('@shikijs/langs/fsharp'))
  },
  {
    "id": "gdresource",
    "name": "GDResource",
    "import": (() => import('@shikijs/langs/gdresource'))
  },
  {
    "id": "gdscript",
    "name": "GDScript",
    "import": (() => import('@shikijs/langs/gdscript'))
  },
  {
    "id": "gdshader",
    "name": "GDShader",
    "import": (() => import('@shikijs/langs/gdshader'))
  },
  {
    "id": "genie",
    "name": "Genie",
    "import": (() => import('@shikijs/langs/genie'))
  },
  {
    "id": "gherkin",
    "name": "Gherkin",
    "import": (() => import('@shikijs/langs/gherkin'))
  },
  {
    "id": "git-commit",
    "name": "Git Commit Message",
    "import": (() => import('@shikijs/langs/git-commit'))
  },
  {
    "id": "git-rebase",
    "name": "Git Rebase Message",
    "import": (() => import('@shikijs/langs/git-rebase'))
  },
  {
    "id": "gleam",
    "name": "Gleam",
    "import": (() => import('@shikijs/langs/gleam'))
  },
  {
    "id": "glimmer-js",
    "name": "Glimmer JS",
    "aliases": [
      "gjs"
    ],
    "import": (() => import('@shikijs/langs/glimmer-js'))
  },
  {
    "id": "glimmer-ts",
    "name": "Glimmer TS",
    "aliases": [
      "gts"
    ],
    "import": (() => import('@shikijs/langs/glimmer-ts'))
  },
  {
    "id": "glsl",
    "name": "GLSL",
    "import": (() => import('@shikijs/langs/glsl'))
  },
  {
    "id": "gnuplot",
    "name": "Gnuplot",
    "import": (() => import('@shikijs/langs/gnuplot'))
  },
  {
    "id": "go",
    "name": "Go",
    "import": (() => import('@shikijs/langs/go'))
  },
  {
    "id": "graphql",
    "name": "GraphQL",
    "aliases": [
      "gql"
    ],
    "import": (() => import('@shikijs/langs/graphql'))
  },
  {
    "id": "groovy",
    "name": "Groovy",
    "import": (() => import('@shikijs/langs/groovy'))
  },
  {
    "id": "hack",
    "name": "Hack",
    "import": (() => import('@shikijs/langs/hack'))
  },
  {
    "id": "haml",
    "name": "Ruby Haml",
    "import": (() => import('@shikijs/langs/haml'))
  },
  {
    "id": "handlebars",
    "name": "Handlebars",
    "aliases": [
      "hbs"
    ],
    "import": (() => import('@shikijs/langs/handlebars'))
  },
  {
    "id": "haskell",
    "name": "Haskell",
    "aliases": [
      "hs"
    ],
    "import": (() => import('@shikijs/langs/haskell'))
  },
  {
    "id": "haxe",
    "name": "Haxe",
    "import": (() => import('@shikijs/langs/haxe'))
  },
  {
    "id": "hcl",
    "name": "HashiCorp HCL",
    "import": (() => import('@shikijs/langs/hcl'))
  },
  {
    "id": "hjson",
    "name": "Hjson",
    "import": (() => import('@shikijs/langs/hjson'))
  },
  {
    "id": "hlsl",
    "name": "HLSL",
    "import": (() => import('@shikijs/langs/hlsl'))
  },
  {
    "id": "html",
    "name": "HTML",
    "import": (() => import('@shikijs/langs/html'))
  },
  {
    "id": "html-derivative",
    "name": "HTML (Derivative)",
    "import": (() => import('@shikijs/langs/html-derivative'))
  },
  {
    "id": "http",
    "name": "HTTP",
    "import": (() => import('@shikijs/langs/http'))
  },
  {
    "id": "hurl",
    "name": "Hurl",
    "import": (() => import('@shikijs/langs/hurl'))
  },
  {
    "id": "hxml",
    "name": "HXML",
    "import": (() => import('@shikijs/langs/hxml'))
  },
  {
    "id": "hy",
    "name": "Hy",
    "import": (() => import('@shikijs/langs/hy'))
  },
  {
    "id": "imba",
    "name": "Imba",
    "import": (() => import('@shikijs/langs/imba'))
  },
  {
    "id": "ini",
    "name": "INI",
    "aliases": [
      "properties"
    ],
    "import": (() => import('@shikijs/langs/ini'))
  },
  {
    "id": "java",
    "name": "Java",
    "import": (() => import('@shikijs/langs/java'))
  },
  {
    "id": "javascript",
    "name": "JavaScript",
    "aliases": [
      "js",
      "cjs",
      "mjs"
    ],
    "import": (() => import('@shikijs/langs/javascript'))
  },
  {
    "id": "jinja",
    "name": "Jinja",
    "import": (() => import('@shikijs/langs/jinja'))
  },
  {
    "id": "jison",
    "name": "Jison",
    "import": (() => import('@shikijs/langs/jison'))
  },
  {
    "id": "json",
    "name": "JSON",
    "import": (() => import('@shikijs/langs/json'))
  },
  {
    "id": "json5",
    "name": "JSON5",
    "import": (() => import('@shikijs/langs/json5'))
  },
  {
    "id": "jsonc",
    "name": "JSON with Comments",
    "import": (() => import('@shikijs/langs/jsonc'))
  },
  {
    "id": "jsonl",
    "name": "JSON Lines",
    "import": (() => import('@shikijs/langs/jsonl'))
  },
  {
    "id": "jsonnet",
    "name": "Jsonnet",
    "import": (() => import('@shikijs/langs/jsonnet'))
  },
  {
    "id": "jssm",
    "name": "JSSM",
    "aliases": [
      "fsl"
    ],
    "import": (() => import('@shikijs/langs/jssm'))
  },
  {
    "id": "jsx",
    "name": "JSX",
    "import": (() => import('@shikijs/langs/jsx'))
  },
  {
    "id": "julia",
    "name": "Julia",
    "aliases": [
      "jl"
    ],
    "import": (() => import('@shikijs/langs/julia'))
  },
  {
    "id": "kdl",
    "name": "KDL",
    "import": (() => import('@shikijs/langs/kdl'))
  },
  {
    "id": "kotlin",
    "name": "Kotlin",
    "aliases": [
      "kt",
      "kts"
    ],
    "import": (() => import('@shikijs/langs/kotlin'))
  },
  {
    "id": "kusto",
    "name": "Kusto",
    "aliases": [
      "kql"
    ],
    "import": (() => import('@shikijs/langs/kusto'))
  },
  {
    "id": "latex",
    "name": "LaTeX",
    "import": (() => import('@shikijs/langs/latex'))
  },
  {
    "id": "lean",
    "name": "Lean 4",
    "aliases": [
      "lean4"
    ],
    "import": (() => import('@shikijs/langs/lean'))
  },
  {
    "id": "less",
    "name": "Less",
    "import": (() => import('@shikijs/langs/less'))
  },
  {
    "id": "liquid",
    "name": "Liquid",
    "import": (() => import('@shikijs/langs/liquid'))
  },
  {
    "id": "llvm",
    "name": "LLVM IR",
    "import": (() => import('@shikijs/langs/llvm'))
  },
  {
    "id": "log",
    "name": "Log file",
    "import": (() => import('@shikijs/langs/log'))
  },
  {
    "id": "logo",
    "name": "Logo",
    "import": (() => import('@shikijs/langs/logo'))
  },
  {
    "id": "lua",
    "name": "Lua",
    "import": (() => import('@shikijs/langs/lua'))
  },
  {
    "id": "luau",
    "name": "Luau",
    "import": (() => import('@shikijs/langs/luau'))
  },
  {
    "id": "make",
    "name": "Makefile",
    "aliases": [
      "makefile"
    ],
    "import": (() => import('@shikijs/langs/make'))
  },
  {
    "id": "markdown",
    "name": "Markdown",
    "aliases": [
      "md"
    ],
    "import": (() => import('@shikijs/langs/markdown'))
  },
  {
    "id": "marko",
    "name": "Marko",
    "import": (() => import('@shikijs/langs/marko'))
  },
  {
    "id": "matlab",
    "name": "MATLAB",
    "import": (() => import('@shikijs/langs/matlab'))
  },
  {
    "id": "mdc",
    "name": "MDC",
    "import": (() => import('@shikijs/langs/mdc'))
  },
  {
    "id": "mdx",
    "name": "MDX",
    "import": (() => import('@shikijs/langs/mdx'))
  },
  {
    "id": "mermaid",
    "name": "Mermaid",
    "aliases": [
      "mmd"
    ],
    "import": (() => import('@shikijs/langs/mermaid'))
  },
  {
    "id": "mipsasm",
    "name": "MIPS Assembly",
    "aliases": [
      "mips"
    ],
    "import": (() => import('@shikijs/langs/mipsasm'))
  },
  {
    "id": "mojo",
    "name": "Mojo",
    "import": (() => import('@shikijs/langs/mojo'))
  },
  {
    "id": "move",
    "name": "Move",
    "import": (() => import('@shikijs/langs/move'))
  },
  {
    "id": "narrat",
    "name": "Narrat Language",
    "aliases": [
      "nar"
    ],
    "import": (() => import('@shikijs/langs/narrat'))
  },
  {
    "id": "nextflow",
    "name": "Nextflow",
    "aliases": [
      "nf"
    ],
    "import": (() => import('@shikijs/langs/nextflow'))
  },
  {
    "id": "nginx",
    "name": "Nginx",
    "import": (() => import('@shikijs/langs/nginx'))
  },
  {
    "id": "nim",
    "name": "Nim",
    "import": (() => import('@shikijs/langs/nim'))
  },
  {
    "id": "nix",
    "name": "Nix",
    "import": (() => import('@shikijs/langs/nix'))
  },
  {
    "id": "nushell",
    "name": "nushell",
    "aliases": [
      "nu"
    ],
    "import": (() => import('@shikijs/langs/nushell'))
  },
  {
    "id": "objective-c",
    "name": "Objective-C",
    "aliases": [
      "objc"
    ],
    "import": (() => import('@shikijs/langs/objective-c'))
  },
  {
    "id": "objective-cpp",
    "name": "Objective-C++",
    "import": (() => import('@shikijs/langs/objective-cpp'))
  },
  {
    "id": "ocaml",
    "name": "OCaml",
    "import": (() => import('@shikijs/langs/ocaml'))
  },
  {
    "id": "openscad",
    "name": "OpenSCAD",
    "aliases": [
      "scad"
    ],
    "import": (() => import('@shikijs/langs/openscad'))
  },
  {
    "id": "pascal",
    "name": "Pascal",
    "import": (() => import('@shikijs/langs/pascal'))
  },
  {
    "id": "perl",
    "name": "Perl",
    "import": (() => import('@shikijs/langs/perl'))
  },
  {
    "id": "php",
    "name": "PHP",
    "import": (() => import('@shikijs/langs/php'))
  },
  {
    "id": "pkl",
    "name": "Pkl",
    "import": (() => import('@shikijs/langs/pkl'))
  },
  {
    "id": "plsql",
    "name": "PL/SQL",
    "import": (() => import('@shikijs/langs/plsql'))
  },
  {
    "id": "po",
    "name": "Gettext PO",
    "aliases": [
      "pot",
      "potx"
    ],
    "import": (() => import('@shikijs/langs/po'))
  },
  {
    "id": "polar",
    "name": "Polar",
    "import": (() => import('@shikijs/langs/polar'))
  },
  {
    "id": "postcss",
    "name": "PostCSS",
    "import": (() => import('@shikijs/langs/postcss'))
  },
  {
    "id": "powerquery",
    "name": "PowerQuery",
    "import": (() => import('@shikijs/langs/powerquery'))
  },
  {
    "id": "powershell",
    "name": "PowerShell",
    "aliases": [
      "ps",
      "ps1"
    ],
    "import": (() => import('@shikijs/langs/powershell'))
  },
  {
    "id": "prisma",
    "name": "Prisma",
    "import": (() => import('@shikijs/langs/prisma'))
  },
  {
    "id": "prolog",
    "name": "Prolog",
    "import": (() => import('@shikijs/langs/prolog'))
  },
  {
    "id": "proto",
    "name": "Protocol Buffer 3",
    "aliases": [
      "protobuf"
    ],
    "import": (() => import('@shikijs/langs/proto'))
  },
  {
    "id": "pug",
    "name": "Pug",
    "aliases": [
      "jade"
    ],
    "import": (() => import('@shikijs/langs/pug'))
  },
  {
    "id": "puppet",
    "name": "Puppet",
    "import": (() => import('@shikijs/langs/puppet'))
  },
  {
    "id": "purescript",
    "name": "PureScript",
    "import": (() => import('@shikijs/langs/purescript'))
  },
  {
    "id": "python",
    "name": "Python",
    "aliases": [
      "py"
    ],
    "import": (() => import('@shikijs/langs/python'))
  },
  {
    "id": "qml",
    "name": "QML",
    "import": (() => import('@shikijs/langs/qml'))
  },
  {
    "id": "qmldir",
    "name": "QML Directory",
    "import": (() => import('@shikijs/langs/qmldir'))
  },
  {
    "id": "qss",
    "name": "Qt Style Sheets",
    "import": (() => import('@shikijs/langs/qss'))
  },
  {
    "id": "r",
    "name": "R",
    "import": (() => import('@shikijs/langs/r'))
  },
  {
    "id": "racket",
    "name": "Racket",
    "import": (() => import('@shikijs/langs/racket'))
  },
  {
    "id": "raku",
    "name": "Raku",
    "aliases": [
      "perl6"
    ],
    "import": (() => import('@shikijs/langs/raku'))
  },
  {
    "id": "razor",
    "name": "ASP.NET Razor",
    "import": (() => import('@shikijs/langs/razor'))
  },
  {
    "id": "reg",
    "name": "Windows Registry Script",
    "import": (() => import('@shikijs/langs/reg'))
  },
  {
    "id": "regexp",
    "name": "RegExp",
    "aliases": [
      "regex"
    ],
    "import": (() => import('@shikijs/langs/regexp'))
  },
  {
    "id": "rel",
    "name": "Rel",
    "import": (() => import('@shikijs/langs/rel'))
  },
  {
    "id": "riscv",
    "name": "RISC-V",
    "import": (() => import('@shikijs/langs/riscv'))
  },
  {
    "id": "rosmsg",
    "name": "ROS Interface",
    "import": (() => import('@shikijs/langs/rosmsg'))
  },
  {
    "id": "rst",
    "name": "reStructuredText",
    "import": (() => import('@shikijs/langs/rst'))
  },
  {
    "id": "ruby",
    "name": "Ruby",
    "aliases": [
      "rb"
    ],
    "import": (() => import('@shikijs/langs/ruby'))
  },
  {
    "id": "rust",
    "name": "Rust",
    "aliases": [
      "rs"
    ],
    "import": (() => import('@shikijs/langs/rust'))
  },
  {
    "id": "sas",
    "name": "SAS",
    "import": (() => import('@shikijs/langs/sas'))
  },
  {
    "id": "sass",
    "name": "Sass",
    "import": (() => import('@shikijs/langs/sass'))
  },
  {
    "id": "scala",
    "name": "Scala",
    "import": (() => import('@shikijs/langs/scala'))
  },
  {
    "id": "scheme",
    "name": "Scheme",
    "import": (() => import('@shikijs/langs/scheme'))
  },
  {
    "id": "scss",
    "name": "SCSS",
    "import": (() => import('@shikijs/langs/scss'))
  },
  {
    "id": "sdbl",
    "name": "1C (Query)",
    "aliases": [
      "1c-query"
    ],
    "import": (() => import('@shikijs/langs/sdbl'))
  },
  {
    "id": "shaderlab",
    "name": "ShaderLab",
    "aliases": [
      "shader"
    ],
    "import": (() => import('@shikijs/langs/shaderlab'))
  },
  {
    "id": "shellscript",
    "name": "Shell",
    "aliases": [
      "bash",
      "sh",
      "shell",
      "zsh"
    ],
    "import": (() => import('@shikijs/langs/shellscript'))
  },
  {
    "id": "shellsession",
    "name": "Shell Session",
    "aliases": [
      "console"
    ],
    "import": (() => import('@shikijs/langs/shellsession'))
  },
  {
    "id": "smalltalk",
    "name": "Smalltalk",
    "import": (() => import('@shikijs/langs/smalltalk'))
  },
  {
    "id": "solidity",
    "name": "Solidity",
    "import": (() => import('@shikijs/langs/solidity'))
  },
  {
    "id": "soy",
    "name": "Closure Templates",
    "aliases": [
      "closure-templates"
    ],
    "import": (() => import('@shikijs/langs/soy'))
  },
  {
    "id": "sparql",
    "name": "SPARQL",
    "import": (() => import('@shikijs/langs/sparql'))
  },
  {
    "id": "splunk",
    "name": "Splunk Query Language",
    "aliases": [
      "spl"
    ],
    "import": (() => import('@shikijs/langs/splunk'))
  },
  {
    "id": "sql",
    "name": "SQL",
    "import": (() => import('@shikijs/langs/sql'))
  },
  {
    "id": "ssh-config",
    "name": "SSH Config",
    "import": (() => import('@shikijs/langs/ssh-config'))
  },
  {
    "id": "stata",
    "name": "Stata",
    "import": (() => import('@shikijs/langs/stata'))
  },
  {
    "id": "stylus",
    "name": "Stylus",
    "aliases": [
      "styl"
    ],
    "import": (() => import('@shikijs/langs/stylus'))
  },
  {
    "id": "svelte",
    "name": "Svelte",
    "import": (() => import('@shikijs/langs/svelte'))
  },
  {
    "id": "swift",
    "name": "Swift",
    "import": (() => import('@shikijs/langs/swift'))
  },
  {
    "id": "system-verilog",
    "name": "SystemVerilog",
    "import": (() => import('@shikijs/langs/system-verilog'))
  },
  {
    "id": "systemd",
    "name": "Systemd Units",
    "import": (() => import('@shikijs/langs/systemd'))
  },
  {
    "id": "talonscript",
    "name": "TalonScript",
    "aliases": [
      "talon"
    ],
    "import": (() => import('@shikijs/langs/talonscript'))
  },
  {
    "id": "tasl",
    "name": "Tasl",
    "import": (() => import('@shikijs/langs/tasl'))
  },
  {
    "id": "tcl",
    "name": "Tcl",
    "import": (() => import('@shikijs/langs/tcl'))
  },
  {
    "id": "templ",
    "name": "Templ",
    "import": (() => import('@shikijs/langs/templ'))
  },
  {
    "id": "terraform",
    "name": "Terraform",
    "aliases": [
      "tf",
      "tfvars"
    ],
    "import": (() => import('@shikijs/langs/terraform'))
  },
  {
    "id": "tex",
    "name": "TeX",
    "import": (() => import('@shikijs/langs/tex'))
  },
  {
    "id": "toml",
    "name": "TOML",
    "import": (() => import('@shikijs/langs/toml'))
  },
  {
    "id": "ts-tags",
    "name": "TypeScript with Tags",
    "aliases": [
      "lit"
    ],
    "import": (() => import('@shikijs/langs/ts-tags'))
  },
  {
    "id": "tsv",
    "name": "TSV",
    "import": (() => import('@shikijs/langs/tsv'))
  },
  {
    "id": "tsx",
    "name": "TSX",
    "import": (() => import('@shikijs/langs/tsx'))
  },
  {
    "id": "turtle",
    "name": "Turtle",
    "import": (() => import('@shikijs/langs/turtle'))
  },
  {
    "id": "twig",
    "name": "Twig",
    "import": (() => import('@shikijs/langs/twig'))
  },
  {
    "id": "typescript",
    "name": "TypeScript",
    "aliases": [
      "ts",
      "cts",
      "mts"
    ],
    "import": (() => import('@shikijs/langs/typescript'))
  },
  {
    "id": "typespec",
    "name": "TypeSpec",
    "aliases": [
      "tsp"
    ],
    "import": (() => import('@shikijs/langs/typespec'))
  },
  {
    "id": "typst",
    "name": "Typst",
    "aliases": [
      "typ"
    ],
    "import": (() => import('@shikijs/langs/typst'))
  },
  {
    "id": "v",
    "name": "V",
    "import": (() => import('@shikijs/langs/v'))
  },
  {
    "id": "vala",
    "name": "Vala",
    "import": (() => import('@shikijs/langs/vala'))
  },
  {
    "id": "vb",
    "name": "Visual Basic",
    "aliases": [
      "cmd"
    ],
    "import": (() => import('@shikijs/langs/vb'))
  },
  {
    "id": "verilog",
    "name": "Verilog",
    "import": (() => import('@shikijs/langs/verilog'))
  },
  {
    "id": "vhdl",
    "name": "VHDL",
    "import": (() => import('@shikijs/langs/vhdl'))
  },
  {
    "id": "viml",
    "name": "Vim Script",
    "aliases": [
      "vim",
      "vimscript"
    ],
    "import": (() => import('@shikijs/langs/viml'))
  },
  {
    "id": "vue",
    "name": "Vue",
    "import": (() => import('@shikijs/langs/vue'))
  },
  {
    "id": "vue-html",
    "name": "Vue HTML",
    "import": (() => import('@shikijs/langs/vue-html'))
  },
  {
    "id": "vue-vine",
    "name": "Vue Vine",
    "import": (() => import('@shikijs/langs/vue-vine'))
  },
  {
    "id": "vyper",
    "name": "Vyper",
    "aliases": [
      "vy"
    ],
    "import": (() => import('@shikijs/langs/vyper'))
  },
  {
    "id": "wasm",
    "name": "WebAssembly",
    "import": (() => import('@shikijs/langs/wasm'))
  },
  {
    "id": "wenyan",
    "name": "Wenyan",
    "aliases": [
      "\u6587\u8A00"
    ],
    "import": (() => import('@shikijs/langs/wenyan'))
  },
  {
    "id": "wgsl",
    "name": "WGSL",
    "import": (() => import('@shikijs/langs/wgsl'))
  },
  {
    "id": "wikitext",
    "name": "Wikitext",
    "aliases": [
      "mediawiki",
      "wiki"
    ],
    "import": (() => import('@shikijs/langs/wikitext'))
  },
  {
    "id": "wit",
    "name": "WebAssembly Interface Types",
    "import": (() => import('@shikijs/langs/wit'))
  },
  {
    "id": "wolfram",
    "name": "Wolfram",
    "aliases": [
      "wl"
    ],
    "import": (() => import('@shikijs/langs/wolfram'))
  },
  {
    "id": "xml",
    "name": "XML",
    "import": (() => import('@shikijs/langs/xml'))
  },
  {
    "id": "xsl",
    "name": "XSL",
    "import": (() => import('@shikijs/langs/xsl'))
  },
  {
    "id": "yaml",
    "name": "YAML",
    "aliases": [
      "yml"
    ],
    "import": (() => import('@shikijs/langs/yaml'))
  },
  {
    "id": "zenscript",
    "name": "ZenScript",
    "import": (() => import('@shikijs/langs/zenscript'))
  },
  {
    "id": "zig",
    "name": "Zig",
    "import": (() => import('@shikijs/langs/zig'))
  }
];
const bundledLanguagesBase = Object.fromEntries(bundledLanguagesInfo.map((i) => [i.id, i.import]));
const bundledLanguagesAlias = Object.fromEntries(bundledLanguagesInfo.flatMap((i) => i.aliases?.map((a) => [a, i.import]) || []));
const bundledLanguages = {
  ...bundledLanguagesBase,
  ...bundledLanguagesAlias
};

const bundledThemesInfo = [
  {
    "id": "andromeeda",
    "displayName": "Andromeeda",
    "type": "dark",
    "import": (() => import('@shikijs/themes/andromeeda'))
  },
  {
    "id": "aurora-x",
    "displayName": "Aurora X",
    "type": "dark",
    "import": (() => import('@shikijs/themes/aurora-x'))
  },
  {
    "id": "ayu-dark",
    "displayName": "Ayu Dark",
    "type": "dark",
    "import": (() => import('@shikijs/themes/ayu-dark'))
  },
  {
    "id": "catppuccin-frappe",
    "displayName": "Catppuccin Frapp\xE9",
    "type": "dark",
    "import": (() => import('@shikijs/themes/catppuccin-frappe'))
  },
  {
    "id": "catppuccin-latte",
    "displayName": "Catppuccin Latte",
    "type": "light",
    "import": (() => import('@shikijs/themes/catppuccin-latte'))
  },
  {
    "id": "catppuccin-macchiato",
    "displayName": "Catppuccin Macchiato",
    "type": "dark",
    "import": (() => import('@shikijs/themes/catppuccin-macchiato'))
  },
  {
    "id": "catppuccin-mocha",
    "displayName": "Catppuccin Mocha",
    "type": "dark",
    "import": (() => import('@shikijs/themes/catppuccin-mocha'))
  },
  {
    "id": "dark-plus",
    "displayName": "Dark Plus",
    "type": "dark",
    "import": (() => import('@shikijs/themes/dark-plus'))
  },
  {
    "id": "dracula",
    "displayName": "Dracula Theme",
    "type": "dark",
    "import": (() => import('@shikijs/themes/dracula'))
  },
  {
    "id": "dracula-soft",
    "displayName": "Dracula Theme Soft",
    "type": "dark",
    "import": (() => import('@shikijs/themes/dracula-soft'))
  },
  {
    "id": "everforest-dark",
    "displayName": "Everforest Dark",
    "type": "dark",
    "import": (() => import('@shikijs/themes/everforest-dark'))
  },
  {
    "id": "everforest-light",
    "displayName": "Everforest Light",
    "type": "light",
    "import": (() => import('@shikijs/themes/everforest-light'))
  },
  {
    "id": "github-dark",
    "displayName": "GitHub Dark",
    "type": "dark",
    "import": (() => import('@shikijs/themes/github-dark'))
  },
  {
    "id": "github-dark-default",
    "displayName": "GitHub Dark Default",
    "type": "dark",
    "import": (() => import('@shikijs/themes/github-dark-default'))
  },
  {
    "id": "github-dark-dimmed",
    "displayName": "GitHub Dark Dimmed",
    "type": "dark",
    "import": (() => import('@shikijs/themes/github-dark-dimmed'))
  },
  {
    "id": "github-dark-high-contrast",
    "displayName": "GitHub Dark High Contrast",
    "type": "dark",
    "import": (() => import('@shikijs/themes/github-dark-high-contrast'))
  },
  {
    "id": "github-light",
    "displayName": "GitHub Light",
    "type": "light",
    "import": (() => import('@shikijs/themes/github-light'))
  },
  {
    "id": "github-light-default",
    "displayName": "GitHub Light Default",
    "type": "light",
    "import": (() => import('@shikijs/themes/github-light-default'))
  },
  {
    "id": "github-light-high-contrast",
    "displayName": "GitHub Light High Contrast",
    "type": "light",
    "import": (() => import('@shikijs/themes/github-light-high-contrast'))
  },
  {
    "id": "gruvbox-dark-hard",
    "displayName": "Gruvbox Dark Hard",
    "type": "dark",
    "import": (() => import('@shikijs/themes/gruvbox-dark-hard'))
  },
  {
    "id": "gruvbox-dark-medium",
    "displayName": "Gruvbox Dark Medium",
    "type": "dark",
    "import": (() => import('@shikijs/themes/gruvbox-dark-medium'))
  },
  {
    "id": "gruvbox-dark-soft",
    "displayName": "Gruvbox Dark Soft",
    "type": "dark",
    "import": (() => import('@shikijs/themes/gruvbox-dark-soft'))
  },
  {
    "id": "gruvbox-light-hard",
    "displayName": "Gruvbox Light Hard",
    "type": "light",
    "import": (() => import('@shikijs/themes/gruvbox-light-hard'))
  },
  {
    "id": "gruvbox-light-medium",
    "displayName": "Gruvbox Light Medium",
    "type": "light",
    "import": (() => import('@shikijs/themes/gruvbox-light-medium'))
  },
  {
    "id": "gruvbox-light-soft",
    "displayName": "Gruvbox Light Soft",
    "type": "light",
    "import": (() => import('@shikijs/themes/gruvbox-light-soft'))
  },
  {
    "id": "houston",
    "displayName": "Houston",
    "type": "dark",
    "import": (() => import('@shikijs/themes/houston'))
  },
  {
    "id": "kanagawa-dragon",
    "displayName": "Kanagawa Dragon",
    "type": "dark",
    "import": (() => import('@shikijs/themes/kanagawa-dragon'))
  },
  {
    "id": "kanagawa-lotus",
    "displayName": "Kanagawa Lotus",
    "type": "light",
    "import": (() => import('@shikijs/themes/kanagawa-lotus'))
  },
  {
    "id": "kanagawa-wave",
    "displayName": "Kanagawa Wave",
    "type": "dark",
    "import": (() => import('@shikijs/themes/kanagawa-wave'))
  },
  {
    "id": "laserwave",
    "displayName": "LaserWave",
    "type": "dark",
    "import": (() => import('@shikijs/themes/laserwave'))
  },
  {
    "id": "light-plus",
    "displayName": "Light Plus",
    "type": "light",
    "import": (() => import('@shikijs/themes/light-plus'))
  },
  {
    "id": "material-theme",
    "displayName": "Material Theme",
    "type": "dark",
    "import": (() => import('@shikijs/themes/material-theme'))
  },
  {
    "id": "material-theme-darker",
    "displayName": "Material Theme Darker",
    "type": "dark",
    "import": (() => import('@shikijs/themes/material-theme-darker'))
  },
  {
    "id": "material-theme-lighter",
    "displayName": "Material Theme Lighter",
    "type": "light",
    "import": (() => import('@shikijs/themes/material-theme-lighter'))
  },
  {
    "id": "material-theme-ocean",
    "displayName": "Material Theme Ocean",
    "type": "dark",
    "import": (() => import('@shikijs/themes/material-theme-ocean'))
  },
  {
    "id": "material-theme-palenight",
    "displayName": "Material Theme Palenight",
    "type": "dark",
    "import": (() => import('@shikijs/themes/material-theme-palenight'))
  },
  {
    "id": "min-dark",
    "displayName": "Min Dark",
    "type": "dark",
    "import": (() => import('@shikijs/themes/min-dark'))
  },
  {
    "id": "min-light",
    "displayName": "Min Light",
    "type": "light",
    "import": (() => import('@shikijs/themes/min-light'))
  },
  {
    "id": "monokai",
    "displayName": "Monokai",
    "type": "dark",
    "import": (() => import('@shikijs/themes/monokai'))
  },
  {
    "id": "night-owl",
    "displayName": "Night Owl",
    "type": "dark",
    "import": (() => import('@shikijs/themes/night-owl'))
  },
  {
    "id": "nord",
    "displayName": "Nord",
    "type": "dark",
    "import": (() => import('@shikijs/themes/nord'))
  },
  {
    "id": "one-dark-pro",
    "displayName": "One Dark Pro",
    "type": "dark",
    "import": (() => import('@shikijs/themes/one-dark-pro'))
  },
  {
    "id": "one-light",
    "displayName": "One Light",
    "type": "light",
    "import": (() => import('@shikijs/themes/one-light'))
  },
  {
    "id": "plastic",
    "displayName": "Plastic",
    "type": "dark",
    "import": (() => import('@shikijs/themes/plastic'))
  },
  {
    "id": "poimandres",
    "displayName": "Poimandres",
    "type": "dark",
    "import": (() => import('@shikijs/themes/poimandres'))
  },
  {
    "id": "red",
    "displayName": "Red",
    "type": "dark",
    "import": (() => import('@shikijs/themes/red'))
  },
  {
    "id": "rose-pine",
    "displayName": "Ros\xE9 Pine",
    "type": "dark",
    "import": (() => import('@shikijs/themes/rose-pine'))
  },
  {
    "id": "rose-pine-dawn",
    "displayName": "Ros\xE9 Pine Dawn",
    "type": "light",
    "import": (() => import('@shikijs/themes/rose-pine-dawn'))
  },
  {
    "id": "rose-pine-moon",
    "displayName": "Ros\xE9 Pine Moon",
    "type": "dark",
    "import": (() => import('@shikijs/themes/rose-pine-moon'))
  },
  {
    "id": "slack-dark",
    "displayName": "Slack Dark",
    "type": "dark",
    "import": (() => import('@shikijs/themes/slack-dark'))
  },
  {
    "id": "slack-ochin",
    "displayName": "Slack Ochin",
    "type": "light",
    "import": (() => import('@shikijs/themes/slack-ochin'))
  },
  {
    "id": "snazzy-light",
    "displayName": "Snazzy Light",
    "type": "light",
    "import": (() => import('@shikijs/themes/snazzy-light'))
  },
  {
    "id": "solarized-dark",
    "displayName": "Solarized Dark",
    "type": "dark",
    "import": (() => import('@shikijs/themes/solarized-dark'))
  },
  {
    "id": "solarized-light",
    "displayName": "Solarized Light",
    "type": "light",
    "import": (() => import('@shikijs/themes/solarized-light'))
  },
  {
    "id": "synthwave-84",
    "displayName": "Synthwave '84",
    "type": "dark",
    "import": (() => import('@shikijs/themes/synthwave-84'))
  },
  {
    "id": "tokyo-night",
    "displayName": "Tokyo Night",
    "type": "dark",
    "import": (() => import('@shikijs/themes/tokyo-night'))
  },
  {
    "id": "vesper",
    "displayName": "Vesper",
    "type": "dark",
    "import": (() => import('@shikijs/themes/vesper'))
  },
  {
    "id": "vitesse-black",
    "displayName": "Vitesse Black",
    "type": "dark",
    "import": (() => import('@shikijs/themes/vitesse-black'))
  },
  {
    "id": "vitesse-dark",
    "displayName": "Vitesse Dark",
    "type": "dark",
    "import": (() => import('@shikijs/themes/vitesse-dark'))
  },
  {
    "id": "vitesse-light",
    "displayName": "Vitesse Light",
    "type": "light",
    "import": (() => import('@shikijs/themes/vitesse-light'))
  }
];
const bundledThemes = Object.fromEntries(bundledThemesInfo.map((i) => [i.id, i.import]));

async function loadBuiltinWasm() {
    const __dirname = dirname(fileURLToPath(import.meta.url));
    await loadWasm(readFile(__dirname + "/onig.wasm"));
}
function loadWasm(options) {
    return onig.loadWasm(options);
}
function createOnigurumaEngine(options) {
    return onig.createOnigurumaEngine(options);
}

export { bundledLanguages, bundledLanguagesInfo, bundledThemes, bundledThemesInfo, codeToTokensWithThemes, createOnigurumaEngine, createShikiInternal, loadBuiltinWasm, loadWasm };
