import type { BundledTheme as ShikiTheme } from "@gerrit0/mini-shiki";
import type { SortStrategy } from "../sort.js";
import type { EntryPointStrategy } from "../entry-point.js";
import type { ReflectionKind } from "../../models/kind.js";
import { type GlobString, type LogLevel, type NeverIfInternal, type NormalizedPath, type NormalizedPathOrModule, type NormalizedPathOrModuleOrFunction, type TagString } from "#utils";
import type { TranslationProxy } from "../../internationalization/internationalization.js";
import type { Application } from "../../application.js";
/** @enum */
export declare const EmitStrategy: {
    readonly both: "both";
    readonly docs: "docs";
    readonly none: "none";
};
/** @hidden */
export type EmitStrategy = (typeof EmitStrategy)[keyof typeof EmitStrategy];
/**
 * Determines how TypeDoc searches for comments.
 * @enum
 */
export declare const CommentStyle: {
    readonly JSDoc: "jsdoc";
    readonly Block: "block";
    readonly Line: "line";
    readonly All: "all";
};
export type CommentStyle = (typeof CommentStyle)[keyof typeof CommentStyle];
export type OutputSpecification = {
    name: string;
    path: string;
    options?: TypeDocOptions;
};
/**
 * List of option names which, with `entryPointStrategy` set to `packages`
 * should only be set at the root level.
 */
export declare const rootPackageOptions: readonly ["plugin", "packageOptions", "outputs", "out", "html", "json", "pretty", "theme", "router", "lightHighlightTheme", "darkHighlightTheme", "highlightLanguages", "ignoredHighlightLanguages", "typePrintWidth", "customCss", "customJs", "customFooterHtml", "customFooterHtmlDisableWrapper", "markdownItOptions", "markdownItLoader", "cname", "favicon", "sourceLinkExternal", "markdownLinkExternal", "lang", "locales", "githubPages", "cacheBust", "hideGenerator", "searchInComments", "searchInDocuments", "cleanOutputDir", "titleLink", "navigationLinks", "sidebarLinks", "navigation", "headings", "sluggerConfiguration", "navigationLeaves", "visibilityFilters", "searchCategoryBoosts", "searchGroupBoosts", "hostedBaseUrl", "useHostedBaseUrlForAbsoluteLinks", "useFirstParagraphOfCommentAsSummary", "includeHierarchySummary", "notRenderedTags", "treatWarningsAsErrors", "treatValidationWarningsAsErrors", "watch", "preserveWatchOutput", "help", "version", "showConfig", "logLevel"];
/**
 * An interface describing all TypeDoc specific options. Generated from a
 * map which contains more information about each option for better types when
 * defining said options.
 * @interface
 */
export type TypeDocOptions = {
    [K in keyof TypeDocOptionMap]?: unknown extends TypeDocOptionMap[K] ? unknown : TypeDocOptionMap[K] extends ManuallyValidatedOption<infer ManuallyValidated> ? ManuallyValidated : TypeDocOptionMap[K] extends NormalizedPath[] | NormalizedPathOrModule[] | NormalizedPathOrModuleOrFunction[] | GlobString[] ? string[] : TypeDocOptionMap[K] extends NormalizedPath ? string : TypeDocOptionMap[K] extends string | string[] | number | boolean ? TypeDocOptionMap[K] : TypeDocOptionMap[K] extends Record<string, boolean> ? Partial<TypeDocOptionMap[K]> | boolean : keyof TypeDocOptionMap[K] | TypeDocOptionMap[K][keyof TypeDocOptionMap[K]];
};
/**
 * Describes all TypeDoc specific options as returned by {@link Options.getValue}, this is
 * slightly more restrictive than the {@link TypeDocOptions} since it does not allow both
 * keys and values for mapped option types, and does not allow partials of flag values.
 * It also does not mark keys as optional.
 * @interface
 */
export type TypeDocOptionValues = {
    [K in keyof TypeDocOptionMap]: unknown extends TypeDocOptionMap[K] ? unknown : TypeDocOptionMap[K] extends ManuallyValidatedOption<infer ManuallyValidated> ? ManuallyValidated : TypeDocOptionMap[K] extends string | string[] | GlobString[] | NormalizedPathOrModule[] | NormalizedPathOrModuleOrFunction[] | number | boolean | Record<string, boolean> ? TypeDocOptionMap[K] : TypeDocOptionMap[K][keyof TypeDocOptionMap[K]];
};
/**
 * Describes TypeDoc options suitable for setting within the `packageOptions` setting.
 *
 * This is a subset of all options specified in {@link TypeDocOptions}.
 */
export interface TypeDocPackageOptions extends Omit<TypeDocOptions, typeof rootPackageOptions[number]> {
}
/**
 * Describes all TypeDoc options. Used internally to provide better types when fetching options.
 * External consumers should likely use {@link TypeDocOptions} instead.
 *
 * If writing a plugin, you may find it useful to use declaration merging to add your options to this interface
 * so that you have autocomplete when using `app.options.getValue`.
 *
 * ```ts
 * declare module "typedoc" {
 *   export interface TypeDocOptionMap {
 *     pluginOption: string[];
 *   }
 * }
 * ```
 */
export interface TypeDocOptionMap {
    options: NormalizedPath;
    tsconfig: NormalizedPath;
    compilerOptions: unknown;
    plugin: NormalizedPathOrModuleOrFunction[];
    lang: string;
    locales: ManuallyValidatedOption<Record<string, Record<string, string>>>;
    packageOptions: ManuallyValidatedOption<TypeDocPackageOptions>;
    entryPoints: GlobString[];
    entryPointStrategy: typeof EntryPointStrategy;
    alwaysCreateEntryPointModule: boolean;
    projectDocuments: GlobString[];
    exclude: GlobString[];
    externalPattern: GlobString[];
    excludeExternals: boolean;
    excludeNotDocumented: boolean;
    excludeNotDocumentedKinds: ReflectionKind.KindString[];
    excludeInternal: boolean;
    excludePrivateClassFields: boolean;
    excludePrivate: boolean;
    excludeProtected: boolean;
    excludeReferences: boolean;
    excludeCategories: string[];
    maxTypeConversionDepth: number;
    name: string;
    includeVersion: boolean;
    disableSources: boolean;
    sourceLinkTemplate: string;
    sourceLinkExternal: boolean;
    markdownLinkExternal: boolean;
    disableGit: boolean;
    gitRevision: string;
    gitRemote: string;
    readme: string;
    basePath: NormalizedPath;
    outputs: ManuallyValidatedOption<Array<OutputSpecification>>;
    out: NormalizedPath;
    html: NormalizedPath;
    json: NormalizedPath;
    pretty: boolean;
    emit: typeof EmitStrategy;
    theme: string;
    router: string;
    lightHighlightTheme: ShikiTheme;
    darkHighlightTheme: ShikiTheme;
    highlightLanguages: string[];
    ignoredHighlightLanguages: string[];
    typePrintWidth: number;
    customCss: NormalizedPath;
    customJs: NormalizedPath;
    markdownItOptions: ManuallyValidatedOption<Record<string, unknown>>;
    /**
     * Will be called when TypeDoc is setting up the markdown parser to use to render markdown.
     * Can be used to add markdown-it plugins to the parser with code like this:
     *
     * ```ts
     * // typedoc.config.mjs
     * import iterator from "markdown-it-for-inline";
     * export default {
     *     /** @param {MarkdownIt} parser *\/
     *     markdownItLoader(parser) {
     *         parser.use(iterator, "foo_replace", "text", function(tokens, idx) {
     *             tokens[idx].content = tokens[idx].content.replace(/foo/g, 'bar');
     *         });
     *     }
     * }
     * ```
     *
     * Note: Unfortunately, markdown-it doesn't ship its own types, so `parser` isn't
     * strictly typed here.
     */
    markdownItLoader: ManuallyValidatedOption<(parser: any) => void>;
    displayBasePath: NormalizedPath;
    cname: string;
    favicon: NormalizedPath;
    githubPages: boolean;
    hostedBaseUrl: string;
    useHostedBaseUrlForAbsoluteLinks: boolean;
    cacheBust: boolean;
    hideGenerator: boolean;
    customFooterHtml: string;
    customFooterHtmlDisableWrapper: boolean;
    searchInComments: boolean;
    searchInDocuments: boolean;
    cleanOutputDir: boolean;
    titleLink: string;
    navigationLinks: ManuallyValidatedOption<Record<string, string>>;
    sidebarLinks: ManuallyValidatedOption<Record<string, string>>;
    navigationLeaves: string[];
    navigation: {
        includeCategories: boolean;
        includeGroups: boolean;
        includeFolders: boolean;
        compactFolders: boolean;
        excludeReferences: boolean;
    };
    headings: {
        readme: boolean;
        document: boolean;
    };
    sluggerConfiguration: {
        lowercase: boolean;
    };
    includeHierarchySummary: boolean;
    visibilityFilters: ManuallyValidatedOption<{
        protected?: boolean;
        private?: boolean;
        inherited?: boolean;
        external?: boolean;
        [tag: TagString]: boolean;
    }>;
    searchCategoryBoosts: ManuallyValidatedOption<Record<string, number>>;
    searchGroupBoosts: ManuallyValidatedOption<Record<string, number>>;
    useFirstParagraphOfCommentAsSummary: boolean;
    commentStyle: typeof CommentStyle;
    useTsLinkResolution: boolean;
    preserveLinkText: boolean;
    jsDocCompatibility: JsDocCompatibility;
    suppressCommentWarningsInDeclarationFiles: boolean;
    blockTags: TagString[];
    inlineTags: TagString[];
    modifierTags: TagString[];
    excludeTags: TagString[];
    notRenderedTags: TagString[];
    externalSymbolLinkMappings: ManuallyValidatedOption<Record<string, Record<string, string>>>;
    cascadedModifierTags: TagString[];
    preservedTypeAnnotationTags: TagString[];
    categorizeByGroup: boolean;
    groupReferencesByType: boolean;
    defaultCategory: string;
    categoryOrder: string[];
    groupOrder: string[];
    sort: SortStrategy[];
    sortEntryPoints: boolean;
    kindSortOrder: ReflectionKind.KindString[];
    treatWarningsAsErrors: boolean;
    treatValidationWarningsAsErrors: boolean;
    intentionallyNotExported: string[];
    validation: ValidationOptions;
    requiredToBeDocumented: ReflectionKind.KindString[];
    packagesRequiringDocumentation: string[];
    intentionallyNotDocumented: string[];
    watch: boolean;
    preserveWatchOutput: boolean;
    help: boolean;
    version: boolean;
    showConfig: boolean;
    logLevel: typeof LogLevel;
    skipErrorChecking: boolean;
}
/**
 * Wrapper type for values in TypeDocOptionMap which are represented with an unknown option type, but
 * have a validation function that checks that they are the given type.
 */
export type ManuallyValidatedOption<T> = {
    __validated: T;
};
export type ValidationOptions = {
    /**
     * If set, TypeDoc will produce warnings when a symbol is referenced by the documentation,
     * but is not included in the documentation.
     */
    notExported: boolean;
    /**
     * If set, TypeDoc will produce warnings about \{\@link\} tags which will produce broken links.
     */
    invalidLink: boolean;
    /**
     * If set, TypeDoc will produce warnings about relative paths within the documentation which could not be resolved.
     */
    invalidPath: boolean;
    /**
     * If set, TypeDoc will produce warnings about \{\@link\} tags which do not link directly to their target.
     */
    rewrittenLink: boolean;
    /**
     * If set, TypeDoc will produce warnings about declarations that do not have doc comments
     */
    notDocumented: boolean;
    /**
     * If set, TypeDoc will produce warnings about `@mergeModuleWith` tags which were not resolved.
     */
    unusedMergeModuleWith: boolean;
};
export type JsDocCompatibility = {
    /**
     * If set, TypeDoc will treat `@example` blocks as code unless they contain a code block.
     * On by default, this is how VSCode renders blocks.
     */
    exampleTag: boolean;
    /**
     * If set, TypeDoc will treat `@default` blocks as code unless they contain a code block.
     * On by default, this is how VSCode renders blocks.
     */
    defaultTag: boolean;
    /**
     * If set, TypeDoc will warn if a `@inheritDoc` tag is spelled without TSDoc capitalization
     * (i.e. `@inheritdoc`). On by default.
     */
    inheritDocTag: boolean;
    /**
     * If set, TypeDoc will not emit warnings about unescaped `{` and `}` characters encountered
     * when parsing a comment. On by default.
     */
    ignoreUnescapedBraces: boolean;
};
/**
 * Converts a given TypeDoc option key to the type of the declaration expected.
 */
export type KeyToDeclaration<K extends keyof TypeDocOptionMap> = TypeDocOptionMap[K] extends boolean ? BooleanDeclarationOption : TypeDocOptionMap[K] extends string | NormalizedPath ? StringDeclarationOption : TypeDocOptionMap[K] extends number ? NumberDeclarationOption : TypeDocOptionMap[K] extends GlobString[] ? GlobArrayDeclarationOption : TypeDocOptionMap[K] extends string[] | NormalizedPath[] | NormalizedPathOrModule[] | NormalizedPathOrModuleOrFunction[] ? ArrayDeclarationOption : unknown extends TypeDocOptionMap[K] ? MixedDeclarationOption | ObjectDeclarationOption : TypeDocOptionMap[K] extends ManuallyValidatedOption<unknown> ? (MixedDeclarationOption & {
    validate(value: unknown, i18n: TranslationProxy): void;
}) | (ObjectDeclarationOption & {
    validate(value: unknown, i18n: TranslationProxy): void;
}) : TypeDocOptionMap[K] extends Record<string, boolean> ? FlagsDeclarationOption<TypeDocOptionMap[K]> : TypeDocOptionMap[K] extends Record<string | number, infer U> ? MapDeclarationOption<U> : never;
export declare enum ParameterHint {
    File = 0,
    Directory = 1
}
export declare enum ParameterType {
    String = 0,
    /**
     * Resolved according to the config directory.
     */
    Path = 1,
    /**
     * Resolved according to the config directory unless it starts with https?://
     */
    UrlOrPath = 2,
    Number = 3,
    Boolean = 4,
    Map = 5,
    Mixed = 6,
    Array = 7,
    /**
     * Resolved according to the config directory.
     */
    PathArray = 8,
    /**
     * Resolved according to the config directory if it starts with `.`
     * @deprecated since 0.28.8, will be removed in 0.29
     */
    ModuleArray = 9,
    /**
     * Resolved according to the config directory if it starts with `.`
     * @internal - only intended for use with the plugin option
     */
    PluginArray = 10,
    /**
     * Relative to the config directory.
     */
    GlobArray = 11,
    /**
     * An object which partially merges user-set values into the defaults.
     */
    Object = 12,
    /**
     * An object with true/false flags
     */
    Flags = 13
}
export interface DeclarationOptionBase {
    /**
     * The option name.
     */
    name: string;
    /**
     * The help text to be displayed to the user when --help is passed.
     *
     * This may be a string, which will be presented directly, or a function,
     * which will be called so that option help can be translated into the user specified locale.
     */
    help: NeverIfInternal<string> | (() => string);
    /**
     * The parameter type, used to convert user configuration values into the expected type.
     * If not set, the type will be a string.
     */
    type?: ParameterType;
    /**
     * If set, this option will be omitted from `--help`, and attempting to specify it on the command
     * line will produce an error.
     */
    configFileOnly?: boolean;
}
export interface StringDeclarationOption extends DeclarationOptionBase {
    /**
     * Specifies the resolution strategy. If `Path` is provided, values will be resolved according to their
     * location in a file. If `String` or no value is provided, values will not be resolved.
     */
    type?: ParameterType.String | ParameterType.Path | ParameterType.UrlOrPath;
    /**
     * If not specified defaults to the empty string for all types.
     */
    defaultValue?: string;
    /**
     * An optional hint for the type of input expected, will be displayed in the help output.
     */
    hint?: ParameterHint;
    /**
     * If specified, when this output is specified TypeDoc will automatically add
     * an output to the `outputs` option whose name is the value of this property with
     * the path set to the value of this option. Should only be used with `type`
     * set to {@link ParameterType.Path}.
     *
     * If any output shortcuts are set, the `outputs` option will be ignored.
     */
    outputShortcut?: string;
    /**
     * An optional validation function that validates a potential value of this option.
     * The function must throw an Error if the validation fails and should do nothing otherwise.
     */
    validate?: (value: string) => void;
}
export interface NumberDeclarationOption extends DeclarationOptionBase {
    type: ParameterType.Number;
    /**
     * Lowest possible value.
     */
    minValue?: number;
    /**
     * Highest possible value.
     */
    maxValue?: number;
    /**
     * If not specified defaults to 0.
     */
    defaultValue?: number;
    /**
     * An optional validation function that validates a potential value of this option.
     * The function must throw an Error if the validation fails and should do nothing otherwise.
     */
    validate?: (value: number) => void;
}
export interface BooleanDeclarationOption extends DeclarationOptionBase {
    type: ParameterType.Boolean;
    /**
     * If not specified defaults to false.
     */
    defaultValue?: boolean;
}
export interface ArrayDeclarationOption extends DeclarationOptionBase {
    type: ParameterType.Array | ParameterType.PathArray | ParameterType.ModuleArray | ParameterType.PluginArray;
    /**
     * If not specified defaults to an empty array.
     */
    defaultValue?: readonly string[];
    /**
     * An optional validation function that validates a potential value of this option.
     * The function must throw an Error if the validation fails and should do nothing otherwise.
     */
    validate?: (value: string[]) => void;
}
export interface GlobArrayDeclarationOption extends DeclarationOptionBase {
    type: ParameterType.GlobArray;
    /**
     * If not specified defaults to an empty array.
     * If specified, globs are relative to cwd when TypeDoc is run.
     */
    defaultValue?: readonly string[];
    /**
     * An optional validation function that validates a potential value of this option.
     * The function must throw an Error if the validation fails and should do nothing otherwise.
     */
    validate?: (value: GlobString[]) => void;
}
export interface MixedDeclarationOption extends DeclarationOptionBase {
    type: ParameterType.Mixed;
    /**
     * If not specified defaults to undefined.
     */
    defaultValue?: unknown;
    /**
     * An optional validation function that validates a potential value of this option.
     * The function must throw an Error if the validation fails and should do nothing otherwise.
     */
    validate?: (value: unknown) => void;
}
export interface ObjectDeclarationOption extends DeclarationOptionBase {
    type: ParameterType.Object;
    /**
     * If not specified defaults to undefined.
     */
    defaultValue?: unknown;
    /**
     * An optional validation function that validates a potential value of this option.
     * The function must throw an Error if the validation fails and should do nothing otherwise.
     */
    validate?: (value: unknown) => void;
}
export interface MapDeclarationOption<T> extends DeclarationOptionBase {
    type: ParameterType.Map;
    /**
     * Maps a given value to the option type. The map type may be a TypeScript enum.
     * In that case, when generating an error message for a mismatched key, the numeric
     * keys will not be listed.
     */
    map: Map<string, T> | Record<string | number, T>;
    /**
     * Unlike the rest of the option types, there is no sensible generic default for mapped option types.
     * The default value for a mapped type must be specified.
     */
    defaultValue: T;
}
export interface FlagsDeclarationOption<T extends Record<string, boolean>> extends DeclarationOptionBase {
    type: ParameterType.Flags;
    /**
     * All of the possible flags, with their default values set.
     */
    defaults: T;
}
export type DeclarationOption = StringDeclarationOption | NumberDeclarationOption | BooleanDeclarationOption | MixedDeclarationOption | ObjectDeclarationOption | MapDeclarationOption<unknown> | ArrayDeclarationOption | GlobArrayDeclarationOption | FlagsDeclarationOption<Record<string, boolean>>;
export interface ParameterTypeToOptionTypeMap {
    [ParameterType.String]: string;
    [ParameterType.Path]: NormalizedPath;
    [ParameterType.UrlOrPath]: NormalizedPath | string;
    [ParameterType.Number]: number;
    [ParameterType.Boolean]: boolean;
    [ParameterType.Mixed]: unknown;
    [ParameterType.Object]: unknown;
    [ParameterType.Array]: string[];
    [ParameterType.PathArray]: NormalizedPath[];
    [ParameterType.ModuleArray]: NormalizedPathOrModule[];
    [ParameterType.PluginArray]: Array<NormalizedPathOrModule | ((app: Application) => void | Promise<void>)>;
    [ParameterType.GlobArray]: GlobString[];
    [ParameterType.Flags]: Record<string, boolean>;
    [ParameterType.Map]: unknown;
}
export type DeclarationOptionToOptionType<T extends DeclarationOption> = T extends MapDeclarationOption<infer U> ? U : T extends FlagsDeclarationOption<infer U> ? U : ParameterTypeToOptionTypeMap[Exclude<T["type"], undefined>];
/**
 * The default conversion function used by the Options container. Readers may
 * re-use this conversion function or implement their own. The arguments reader
 * implements its own since 'false' should not be converted to true for a boolean option.
 * @param value The value to convert.
 * @param option The option for which the value should be converted.
 * @returns The result of the conversion. Might be the value or an error.
 */
export declare function convert(value: unknown, option: DeclarationOption, configPath: string, oldValue?: unknown): unknown;
export declare function getDefaultValue(option: DeclarationOption): unknown;
