/*
 * Decompiled with CFR 0.152.
 */
package org.forgerock.json.jose.jws.handlers;

import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.Signature;
import java.security.SignatureException;
import java.security.interfaces.ECKey;
import java.security.interfaces.ECPrivateKey;
import java.security.interfaces.ECPublicKey;
import java.util.Arrays;
import org.forgerock.json.jose.exceptions.JwsException;
import org.forgerock.json.jose.exceptions.JwsSigningException;
import org.forgerock.json.jose.jws.JwsAlgorithm;
import org.forgerock.json.jose.jws.JwsAlgorithmType;
import org.forgerock.json.jose.jws.SupportedEllipticCurve;
import org.forgerock.json.jose.jws.handlers.SigningHandler;
import org.forgerock.json.jose.utils.DerUtils;
import org.forgerock.json.jose.utils.Utils;
import org.forgerock.util.Reject;

public class ECDSASigningHandler
implements SigningHandler {
    private final ECPrivateKey signingKey;
    private final ECPublicKey verificationKey;
    private final SupportedEllipticCurve curve;

    public ECDSASigningHandler(ECPrivateKey signingKey) {
        this.signingKey = signingKey;
        this.verificationKey = null;
        this.curve = this.validateKey(signingKey);
    }

    public ECDSASigningHandler(ECPublicKey verificationKey) {
        this.signingKey = null;
        this.verificationKey = verificationKey;
        this.curve = this.validateKey(verificationKey);
    }

    @Override
    public byte[] sign(JwsAlgorithm algorithm, String data) {
        return this.sign(algorithm, data.getBytes(Utils.CHARSET));
    }

    @Override
    public byte[] sign(JwsAlgorithm algorithm, byte[] data) {
        this.validateAlgorithm(algorithm);
        try {
            Signature signature = Signature.getInstance(algorithm.getAlgorithm());
            signature.initSign(this.signingKey);
            signature.update(data);
            return ECDSASigningHandler.derDecode(signature.sign(), this.curve.getSignatureSize());
        }
        catch (InvalidKeyException | SignatureException e) {
            throw new JwsSigningException(e);
        }
        catch (NoSuchAlgorithmException e) {
            throw new JwsSigningException("Unsupported Signing Algorithm, " + algorithm.getAlgorithm(), e);
        }
    }

    @Override
    public boolean verify(JwsAlgorithm algorithm, byte[] data, byte[] signature) {
        this.validateAlgorithm(algorithm);
        try {
            Signature validator = Signature.getInstance(algorithm.getAlgorithm());
            validator.initVerify(this.verificationKey);
            validator.update(data);
            return validator.verify(ECDSASigningHandler.derEncode(signature));
        }
        catch (InvalidKeyException | SignatureException e) {
            throw new JwsSigningException(e);
        }
        catch (NoSuchAlgorithmException e) {
            throw new JwsSigningException("Unsupported Signing Algorithm, " + algorithm.getAlgorithm(), e);
        }
    }

    private void validateAlgorithm(JwsAlgorithm algorithm) {
        Reject.ifNull((Object)algorithm, (String)"Algorithm must not be null.");
        Reject.ifTrue((algorithm.getAlgorithmType() != JwsAlgorithmType.ECDSA ? 1 : 0) != 0, (String)"Not an ECDSA algorithm.");
    }

    private SupportedEllipticCurve validateKey(ECKey key) {
        Reject.ifNull((Object)key);
        try {
            return SupportedEllipticCurve.forKey(key);
        }
        catch (IllegalArgumentException ex) {
            throw new JwsException(ex);
        }
    }

    private static byte[] derDecode(byte[] signature, int signatureSize) {
        ByteBuffer buffer = ByteBuffer.wrap(signature);
        if (buffer.get() != 48) {
            throw new JwsSigningException("Unable to decode DER signature");
        }
        DerUtils.readLength(buffer);
        byte[] output = new byte[signatureSize];
        int componentSize = signatureSize >> 1;
        DerUtils.readUnsignedInteger(buffer, output, 0, componentSize);
        DerUtils.readUnsignedInteger(buffer, output, componentSize, componentSize);
        return output;
    }

    private static byte[] derEncode(byte[] signature) {
        Reject.ifNull((Object)signature);
        SupportedEllipticCurve curve = SupportedEllipticCurve.forSignature(signature);
        int midPoint = curve.getSignatureSize() >> 1;
        BigInteger r = new BigInteger(Arrays.copyOfRange(signature, 0, midPoint));
        BigInteger s = new BigInteger(Arrays.copyOfRange(signature, midPoint, signature.length));
        ByteBuffer params = ByteBuffer.allocate(signature.length + 6);
        DerUtils.writeInteger(params, r.toByteArray());
        DerUtils.writeInteger(params, s.toByteArray());
        int size = params.position();
        ByteBuffer sequence = ByteBuffer.allocate(size + 5);
        sequence.put((byte)48);
        DerUtils.writeLength(sequence, size);
        sequence.put(params.flip());
        byte[] result = new byte[sequence.position()];
        sequence.flip().get(result);
        return result;
    }
}

