/*
 * Decompiled with CFR 0.152.
 */
package org.forgerock.audit.events.handlers.writers;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.nio.charset.StandardCharsets;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.zip.GZIPOutputStream;
import org.forgerock.audit.events.handlers.FileBasedEventHandlerConfiguration;
import org.forgerock.audit.events.handlers.writers.MeteredStream;
import org.forgerock.audit.events.handlers.writers.TextWriter;
import org.forgerock.audit.retention.FileNamingPolicy;
import org.forgerock.audit.retention.RetentionPolicy;
import org.forgerock.audit.rotation.RotatableObject;
import org.forgerock.audit.rotation.RotationContext;
import org.forgerock.audit.rotation.RotationHooks;
import org.forgerock.audit.rotation.RotationPolicy;
import org.forgerock.util.annotations.VisibleForTesting;
import org.forgerock.util.time.Duration;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RotatableWriter
implements TextWriter,
RotatableObject {
    private static final Logger logger = LoggerFactory.getLogger(RotatableWriter.class);
    private static final Duration FIVE_SECONDS = Duration.duration((String)"5s");
    private final List<RotationPolicy> rotationPolicies;
    private final List<RetentionPolicy> retentionPolicies;
    private final FileNamingPolicy fileNamingPolicy;
    private ScheduledExecutorService rotator;
    private DateTime lastRotationTime;
    private final boolean rotationEnabled;
    private final File file;
    private RotationHooks rotationHooks = new RotationHooks.NoOpRotatationHooks();
    private final AtomicBoolean isRotating = new AtomicBoolean(false);
    private MeteredStream meteredStream;
    private BufferedWriter writer;
    private final ReentrantReadWriteLock readWriteLock = new ReentrantReadWriteLock();
    private final RolloverLifecycleHook rolloverLifecycleHook;
    public static final RolloverLifecycleHook NOOP_ROLLOVER_LIFECYCLE_HOOK = new RolloverLifecycleHook(){

        @Override
        public void beforeRollingOver() {
        }

        @Override
        public void afterRollingOver() {
        }
    };

    public RotatableWriter(File file, FileBasedEventHandlerConfiguration configuration, boolean append) throws IOException {
        this(file, configuration, append, configuration.getFileRotation().buildTimeStampFileNamingPolicy(file));
    }

    public RotatableWriter(File file, FileBasedEventHandlerConfiguration configuration, boolean append, RolloverLifecycleHook rolloverLifecycleHook) throws IOException {
        this(file, configuration, append, configuration.getFileRotation().buildTimeStampFileNamingPolicy(file), rolloverLifecycleHook);
    }

    @VisibleForTesting
    RotatableWriter(File file, FileBasedEventHandlerConfiguration configuration, boolean append, FileNamingPolicy fileNamingPolicy) throws IOException {
        this(file, configuration, append, fileNamingPolicy, NOOP_ROLLOVER_LIFECYCLE_HOOK);
    }

    private RotatableWriter(File file, FileBasedEventHandlerConfiguration configuration, boolean append, FileNamingPolicy fileNamingPolicy, RolloverLifecycleHook rolloverLifecycleHook) throws IOException {
        this.file = file;
        this.fileNamingPolicy = fileNamingPolicy;
        this.rotationEnabled = configuration.getFileRotation().isRotationEnabled();
        long lastModified = file.lastModified();
        this.lastRotationTime = lastModified > 0L ? new DateTime(file.lastModified(), DateTimeZone.UTC) : DateTime.now((DateTimeZone)DateTimeZone.UTC);
        this.rolloverLifecycleHook = rolloverLifecycleHook;
        this.writer = this.constructWriter(file, append);
        this.retentionPolicies = configuration.getFileRetention().buildRetentionPolicies();
        this.rotationPolicies = configuration.getFileRotation().buildRotationPolicies();
        this.scheduleRotationAndRetentionChecks(configuration);
    }

    @Override
    public void rotateIfNeeded() throws IOException {
        if (!this.rotationEnabled || this.isRotating.get()) {
            return;
        }
        this.readWriteLock.writeLock().lock();
        try {
            for (RotationPolicy rotationPolicy : this.rotationPolicies) {
                if (!rotationPolicy.shouldRotateFile(this)) continue;
                if (logger.isTraceEnabled()) {
                    logger.trace("Must rotate: {}", (Object)this.file.getAbsolutePath());
                }
                this.isRotating.set(true);
                if (this.rotate() && logger.isTraceEnabled()) {
                    logger.trace("Finished rotation for: {}", (Object)this.file.getAbsolutePath());
                }
                break;
            }
        }
        finally {
            this.readWriteLock.writeLock().unlock();
            this.isRotating.set(false);
        }
    }

    private void deleteFilesIfNeeded() throws IOException {
        this.readWriteLock.writeLock().lock();
        try {
            Set<File> filesToDelete = this.checkRetention();
            if (!filesToDelete.isEmpty()) {
                this.deleteFiles(filesToDelete);
            }
        }
        finally {
            this.readWriteLock.writeLock().unlock();
        }
    }

    private boolean rotate() throws IOException {
        boolean rotationHappened = false;
        RotationContext context = new RotationContext();
        context.setWriter(this.writer);
        File currentFile = this.fileNamingPolicy.getInitialName();
        context.setInitialFile(currentFile);
        if (currentFile.exists()) {
            File newFile = this.fileNamingPolicy.getNextName();
            context.setNextFile(newFile);
            this.rotationHooks.preRotationAction(context);
            this.writer.close();
            if (logger.isTraceEnabled()) {
                logger.trace("Renaming {} to {}", (Object)currentFile.getAbsolutePath(), (Object)newFile.getAbsolutePath());
            }
            if (currentFile.renameTo(new File(newFile.getAbsolutePath().replace(".gz", "")))) {
                rotationHappened = true;
                if (currentFile.createNewFile()) {
                    this.writer = this.constructWriter(currentFile, true);
                    context.setWriter(this.writer);
                    this.rotationHooks.postRotationAction(context);
                    if (newFile.getName().endsWith(".gz")) {
                        byte[] buffer = new byte[1024];
                        try {
                            int bytes_read;
                            FileOutputStream fileOutputStream = new FileOutputStream(newFile);
                            GZIPOutputStream gzipOuputStream = new GZIPOutputStream(fileOutputStream);
                            FileInputStream fileInput = new FileInputStream(newFile.getAbsolutePath().replace(".gz", ""));
                            while ((bytes_read = fileInput.read(buffer)) > 0) {
                                gzipOuputStream.write(buffer, 0, bytes_read);
                            }
                            fileInput.close();
                            gzipOuputStream.finish();
                            gzipOuputStream.close();
                            new File(newFile.getAbsolutePath().replace(".gz", "")).delete();
                        }
                        catch (IOException ex) {
                            logger.error("compression {}: {}", new Object[]{newFile, ex.toString(), ex});
                        }
                    }
                } else {
                    logger.error("Unable to resume writing to audit file {}; further events will not be logged", (Object)currentFile.toString());
                }
            } else {
                logger.error("Unable to rename the audit file {}; further events will continue to be logged to the current file", (Object)currentFile.toString());
                this.writer = this.constructWriter(currentFile, true);
            }
            this.lastRotationTime = DateTime.now((DateTimeZone)DateTimeZone.UTC);
        }
        return rotationHappened;
    }

    private Set<File> checkRetention() throws IOException {
        HashSet<File> filesToDelete = new HashSet<File>();
        for (RetentionPolicy retentionPolicy : this.retentionPolicies) {
            filesToDelete.addAll(retentionPolicy.deleteFiles(this.fileNamingPolicy));
        }
        return filesToDelete;
    }

    private void deleteFiles(Set<File> files) {
        for (File file : files) {
            if (logger.isInfoEnabled()) {
                logger.info("Deleting file {}", (Object)file.getAbsolutePath());
            }
            if (file.delete() || !logger.isWarnEnabled()) continue;
            logger.warn("Could not delete file {}", (Object)file.getAbsolutePath());
        }
    }

    @Override
    public long getBytesWritten() {
        logger.trace("bytes written={}", (Object)this.meteredStream.getBytesWritten());
        return this.meteredStream.getBytesWritten();
    }

    @Override
    public DateTime getLastRotationTime() {
        return this.lastRotationTime;
    }

    @Override
    public void close() throws IOException {
        if (this.rotator != null) {
            boolean interrupted = false;
            this.rotator.shutdown();
            try {
                while (!this.rotator.awaitTermination(500L, TimeUnit.MILLISECONDS)) {
                    logger.debug("Waiting to terminate the rotator thread.");
                }
            }
            catch (InterruptedException ex) {
                logger.error("Unable to terminate the rotator thread", (Throwable)ex);
                interrupted = true;
            }
            finally {
                if (interrupted) {
                    Thread.currentThread().interrupt();
                }
            }
        }
        this.writer.close();
    }

    @Override
    public void shutdown() {
        try {
            this.close();
        }
        catch (IOException e) {
            logger.error("Error when performing shutdown", (Throwable)e);
        }
    }

    @Override
    public void registerRotationHooks(RotationHooks rotationHooks) {
        this.rotationHooks = rotationHooks;
    }

    @Override
    public void write(String str) throws IOException {
        ReentrantReadWriteLock.ReadLock lock = this.readWriteLock.readLock();
        try {
            lock.lock();
            logger.trace("Actually writing to file: {}", (Object)str);
            this.writer.write(str);
        }
        finally {
            lock.unlock();
        }
        this.rotateIfNeeded();
    }

    public boolean forceRotation() throws IOException {
        this.readWriteLock.writeLock().lock();
        try {
            this.isRotating.set(true);
            boolean bl = this.rotate();
            return bl;
        }
        finally {
            this.isRotating.set(false);
            this.readWriteLock.writeLock().unlock();
        }
    }

    @Override
    public void flush() throws IOException {
        this.writer.flush();
    }

    private BufferedWriter constructWriter(File csvFile, boolean append) throws IOException {
        FileOutputStream stream = new FileOutputStream(csvFile, append);
        this.meteredStream = new MeteredStream(stream, this.file.length());
        OutputStreamWriter osw = new OutputStreamWriter((OutputStream)this.meteredStream, StandardCharsets.UTF_8);
        return new BufferedWriter(osw);
    }

    private void scheduleRotationAndRetentionChecks(FileBasedEventHandlerConfiguration configuration) throws IOException {
        Duration rotationCheckInterval = this.parseDuration("rotation and retention check interval", configuration.getRotationRetentionCheckInterval(), FIVE_SECONDS);
        if (!this.rotationPolicies.isEmpty() || !this.retentionPolicies.isEmpty()) {
            if (rotationCheckInterval.isUnlimited() || rotationCheckInterval.isZero()) {
                throw new IOException("Rotation and retention check interval set to an invalid value: " + rotationCheckInterval);
            }
            this.rotator = Executors.newScheduledThreadPool(1);
            this.rotator.scheduleAtFixedRate(new Runnable(){

                @Override
                public void run() {
                    RotatableWriter.this.rolloverLifecycleHook.beforeRollingOver();
                    try {
                        try {
                            RotatableWriter.this.rotateIfNeeded();
                        }
                        catch (Exception e) {
                            logger.error("Failure when applying a rotation policy to file {}", (Object)RotatableWriter.this.fileNamingPolicy.getInitialName(), (Object)e);
                        }
                        try {
                            RotatableWriter.this.deleteFilesIfNeeded();
                        }
                        catch (Exception e) {
                            logger.error("Failure when applying a retention policy to file {}", (Object)RotatableWriter.this.fileNamingPolicy.getInitialName(), (Object)e);
                        }
                    }
                    finally {
                        RotatableWriter.this.rolloverLifecycleHook.afterRollingOver();
                    }
                }
            }, rotationCheckInterval.to(TimeUnit.MILLISECONDS), rotationCheckInterval.to(TimeUnit.MILLISECONDS), TimeUnit.MILLISECONDS);
        }
    }

    private Duration parseDuration(String description, String duration, Duration defaultValue) {
        try {
            return Duration.duration((String)duration);
        }
        catch (IllegalArgumentException e) {
            logger.warn("Invalid {} value: '{}'", (Object)description, (Object)duration);
            return defaultValue;
        }
    }

    @VisibleForTesting
    List<RotationPolicy> getRotationPolicies() {
        return this.rotationPolicies;
    }

    public static interface RolloverLifecycleHook {
        public void beforeRollingOver();

        public void afterRollingOver();
    }
}

