/*
 * Decompiled with CFR 0.152.
 */
package org.forgerock.audit.handlers.json;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.charset.StandardCharsets;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.forgerock.audit.batch.CommonAuditBatchConfiguration;
import org.forgerock.audit.handlers.json.ByteBufferOutputStream;
import org.forgerock.audit.handlers.json.JsonAuditEventHandler;
import org.forgerock.audit.handlers.json.JsonAuditEventHandlerConfiguration;
import org.forgerock.audit.retention.FileNamingPolicy;
import org.forgerock.audit.retention.RetentionPolicy;
import org.forgerock.audit.rotation.RotatableObject;
import org.forgerock.audit.rotation.RotationHooks;
import org.forgerock.audit.rotation.RotationPolicy;
import org.forgerock.audit.util.ElasticsearchUtil;
import org.forgerock.json.JsonValue;
import org.forgerock.util.Reject;
import org.forgerock.util.Utils;
import org.forgerock.util.time.Duration;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class JsonFileWriter {
    private static final Logger logger = LoggerFactory.getLogger(JsonFileWriter.class);
    private static final int MIN_QUEUE_SIZE = 100000;
    static final String LOG_FILE_NAME_SUFFIX = "audit.json";
    private final boolean elasticsearchCompatible;
    private final BlockingQueue<QueueEntry> queue;
    private final ScheduledExecutorService scheduler;
    private final QueueConsumer queueConsumer;
    private final Duration writeInterval;

    JsonFileWriter(Set<String> topics, JsonAuditEventHandlerConfiguration configuration, boolean autoFlush) {
        this.elasticsearchCompatible = configuration.isElasticsearchCompatible();
        this.queue = new ArrayBlockingQueue<QueueEntry>(Math.max(configuration.getBuffering().getMaxSize(), 100000));
        this.scheduler = Executors.newScheduledThreadPool(1, Utils.newThreadFactory(null, (String)"audit-json-%d", (boolean)false));
        this.queueConsumer = new QueueConsumer(LOG_FILE_NAME_SUFFIX, topics, configuration, autoFlush, this.queue, this.scheduler);
        this.writeInterval = this.parseWriteInterval(configuration);
    }

    private Duration parseWriteInterval(JsonAuditEventHandlerConfiguration configuration) {
        Duration writeInterval;
        String writeIntervalString = configuration.getBuffering().getWriteInterval();
        try {
            writeInterval = Duration.duration((String)writeIntervalString);
        }
        catch (Exception e) {
            writeInterval = null;
        }
        if (writeInterval == null || writeInterval.getValue() <= 0L) {
            logger.info("writeInterval '{}' is invalid, so falling back to {}", (Object)writeIntervalString, (Object)CommonAuditBatchConfiguration.POLLING_INTERVAL);
            return CommonAuditBatchConfiguration.POLLING_INTERVAL;
        }
        return writeInterval;
    }

    void startup() {
        this.scheduler.scheduleAtFixedRate(this.queueConsumer, 0L, this.writeInterval.to(TimeUnit.MILLISECONDS), TimeUnit.MILLISECONDS);
    }

    void shutdown() {
        if (!this.scheduler.isShutdown()) {
            this.queueConsumer.shutdown();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void put(String topic, JsonValue event) throws InterruptedException, IOException {
        if (this.elasticsearchCompatible) {
            ElasticsearchUtil.renameField((JsonValue)event, (String)"_id", (String)"_eventId");
            try {
                byte[] bytes = ElasticsearchUtil.normalizeJson((JsonValue)event).getBytes(StandardCharsets.UTF_8);
                this.queue.put(new QueueEntry(topic, bytes));
            }
            finally {
                ElasticsearchUtil.renameField((JsonValue)event, (String)"_eventId", (String)"_id");
            }
        } else {
            this.queue.put(new QueueEntry(topic, JsonAuditEventHandler.OBJECT_MAPPER.writeValueAsBytes(event.getObject())));
        }
    }

    boolean rotateFile(String topic) throws InterruptedException {
        if (this.queueConsumer.isRotationEnabled()) {
            this.queue.put(new QueueEntry(topic, QueueEntry.ROTATE_FILE_ENTRY));
            return true;
        }
        return false;
    }

    void flushFileBuffer(String topic) throws InterruptedException {
        this.queue.put(new QueueEntry(topic, QueueEntry.FLUSH_FILE_ENTRY));
    }

    Path getTopicFilePath(String topic) {
        QueueConsumer.TopicEntry topicEntry = (QueueConsumer.TopicEntry)this.queueConsumer.topicEntryMap.get(topic);
        return topicEntry == null ? null : topicEntry.filePath;
    }

    private static final class QueueConsumer
    implements Runnable {
        private static final int BATCH_SIZE = 5000;
        private static final int OUTPUT_BUF_INITIAL_SIZE = 16384;
        private static final byte[] NEWLINE_UTF_8_BYTES = "\n".getBytes(StandardCharsets.UTF_8);
        private final boolean flushOnShutdown;
        private final boolean rotationEnabled;
        private final boolean hasRotationOrRetentionPolicies;
        private final List<RotationPolicy> rotationPolicies;
        private final List<RetentionPolicy> retentionPolicies;
        private final Set<File> filesToDelete;
        private final BlockingQueue<QueueEntry> queue;
        private final ScheduledExecutorService scheduler;
        private final Map<String, TopicEntry> topicEntryMap;
        private final List<QueueEntry> drainList;
        private volatile boolean shutdown;

        private QueueConsumer(String fileNameSuffix, Set<String> topics, JsonAuditEventHandlerConfiguration configuration, boolean flushOnShutdown, BlockingQueue<QueueEntry> queue, ScheduledExecutorService scheduler) {
            this.queue = queue;
            this.scheduler = scheduler;
            this.flushOnShutdown = flushOnShutdown;
            this.drainList = new ArrayList<QueueEntry>(5000);
            this.rotationEnabled = configuration.getFileRotation().isRotationEnabled();
            this.rotationPolicies = configuration.getFileRotation().buildRotationPolicies();
            this.retentionPolicies = configuration.getFileRetention().buildRetentionPolicies();
            this.hasRotationOrRetentionPolicies = this.rotationEnabled && !this.rotationPolicies.isEmpty() || !this.retentionPolicies.isEmpty();
            this.filesToDelete = new HashSet<File>();
            HashMap<String, TopicEntry> topicEntryMap = new HashMap<String, TopicEntry>();
            for (String topic : topics) {
                String fileName = topic + '.' + fileNameSuffix;
                topicEntryMap.put(topic, new TopicEntry(fileName, configuration));
            }
            this.topicEntryMap = Collections.unmodifiableMap(topicEntryMap);
        }

        void shutdown() {
            if (!this.shutdown) {
                this.shutdown = true;
                if (this.flushOnShutdown) {
                    boolean interrupted = false;
                    while (!this.scheduler.isTerminated()) {
                        try {
                            this.scheduler.awaitTermination(1L, TimeUnit.MINUTES);
                        }
                        catch (InterruptedException e) {
                            interrupted = true;
                        }
                    }
                    if (interrupted) {
                        Thread.currentThread().interrupt();
                    }
                }
            }
        }

        @Override
        public void run() {
            if (!this.shutdown) {
                do {
                    this.writeEvents();
                } while (!this.queue.isEmpty() && !this.shutdown);
            }
            if (this.shutdown) {
                this.scheduler.shutdown();
                try {
                    if (this.flushOnShutdown) {
                        while (!this.queue.isEmpty()) {
                            this.writeEvents();
                        }
                        for (TopicEntry topicEntry : this.topicEntryMap.values()) {
                            topicEntry.flush();
                        }
                    }
                }
                finally {
                    Utils.closeSilently(this.topicEntryMap.values());
                }
            }
        }

        private void writeEvents() {
            this.drainList.clear();
            try {
                int n = this.queue.drainTo(this.drainList, 5000);
                for (int i = 0; i < n; ++i) {
                    QueueEntry entry = this.drainList.get(i);
                    TopicEntry topicEntry = this.topicEntryMap.get(entry.topic);
                    if (topicEntry == null) {
                        logger.warn("Unrecognised topic: " + entry.topic);
                        continue;
                    }
                    if (entry.isRotateEntry()) {
                        topicEntry.rotateNow();
                        continue;
                    }
                    if (entry.isFlushEntry()) {
                        topicEntry.flush();
                        continue;
                    }
                    topicEntry.write(entry.event);
                }
                if (n == 0) {
                    for (TopicEntry topicEntry : this.topicEntryMap.values()) {
                        topicEntry.flush();
                    }
                }
                if (this.hasRotationOrRetentionPolicies) {
                    for (TopicEntry topicEntry : this.topicEntryMap.values()) {
                        topicEntry.rotateIfNeeded();
                    }
                }
            }
            catch (IOException e) {
                logger.error("JSON file write failed", (Throwable)e);
            }
            catch (Exception e) {
                logger.error("Unexpected failure", (Throwable)e);
            }
        }

        boolean isRotationEnabled() {
            return this.rotationEnabled;
        }

        private class TopicEntry
        implements RotatableObject,
        Closeable {
            private static final int FILE_BUFFER_THRESHOLD = 8192;
            private final Path filePath;
            private final FileNamingPolicy fileNamingPolicy;
            private final ByteBufferOutputStream outputStream;
            private DateTime lastRotationTime;
            private FileChannel fileChannel;
            private long positionInFile;

            TopicEntry(String fileName, JsonAuditEventHandlerConfiguration configuration) {
                try {
                    this.outputStream = new ByteBufferOutputStream(ByteBuffer.allocateDirect(16384));
                    Path directoryPath = Paths.get(configuration.getLogDirectory(), new String[0]);
                    if (Files.notExists(directoryPath, new LinkOption[0])) {
                        Files.createDirectory(directoryPath, new FileAttribute[0]);
                    }
                    this.filePath = directoryPath.resolve(fileName);
                    if (Files.notExists(this.filePath, new LinkOption[0])) {
                        this.fileChannel = FileChannel.open(this.filePath, StandardOpenOption.CREATE_NEW, StandardOpenOption.WRITE);
                    } else {
                        this.fileChannel = FileChannel.open(this.filePath, StandardOpenOption.WRITE);
                        this.positionInFile = this.fileChannel.size();
                    }
                    File currentFile = this.filePath.toFile();
                    this.fileNamingPolicy = configuration.getFileRotation().buildTimeStampFileNamingPolicy(currentFile);
                    long lastModified = currentFile.lastModified();
                    this.lastRotationTime = lastModified > 0L ? new DateTime(lastModified, DateTimeZone.UTC) : DateTime.now((DateTimeZone)DateTimeZone.UTC);
                }
                catch (IOException e) {
                    throw new RuntimeException("Failed to create or open file", e);
                }
            }

            void write(byte[] bytes) throws IOException {
                this.outputStream.write(bytes);
                this.outputStream.write(NEWLINE_UTF_8_BYTES);
                if (this.outputStream.byteBuffer().position() >= 8192) {
                    this.outputStream.byteBuffer().flip();
                    try {
                        this.positionInFile += (long)this.fileChannel.write(this.outputStream.byteBuffer(), this.positionInFile);
                    }
                    finally {
                        this.outputStream.clear();
                    }
                }
            }

            void flush() {
                if (this.outputStream.byteBuffer().position() != 0) {
                    this.outputStream.byteBuffer().flip();
                    try {
                        this.positionInFile += (long)this.fileChannel.write(this.outputStream.byteBuffer(), this.positionInFile);
                    }
                    catch (IOException e) {
                        logger.error("Failed to flush file buffer", (Throwable)e);
                    }
                    finally {
                        this.outputStream.clear();
                    }
                }
            }

            public long getBytesWritten() {
                return this.positionInFile;
            }

            public DateTime getLastRotationTime() {
                return this.lastRotationTime;
            }

            public void rotateIfNeeded() throws IOException {
                if (QueueConsumer.this.rotationEnabled && !QueueConsumer.this.rotationPolicies.isEmpty()) {
                    for (RotationPolicy rotationPolicy : QueueConsumer.this.rotationPolicies) {
                        if (!rotationPolicy.shouldRotateFile((RotatableObject)this)) continue;
                        this.rotateNow();
                        break;
                    }
                }
                if (!QueueConsumer.this.retentionPolicies.isEmpty()) {
                    QueueConsumer.this.filesToDelete.clear();
                    for (RetentionPolicy retentionPolicy : QueueConsumer.this.retentionPolicies) {
                        QueueConsumer.this.filesToDelete.addAll(retentionPolicy.deleteFiles(this.fileNamingPolicy));
                    }
                    if (!QueueConsumer.this.filesToDelete.isEmpty()) {
                        for (File file : QueueConsumer.this.filesToDelete) {
                            if (file.delete() || !logger.isWarnEnabled()) continue;
                            logger.warn("Could not delete file {}", (Object)file.getAbsolutePath());
                        }
                    }
                }
            }

            void rotateNow() throws IOException {
                this.fileChannel.close();
                Path archivedFilePath = this.fileNamingPolicy.getNextName().toPath();
                Files.move(this.filePath, archivedFilePath, new CopyOption[0]);
                this.fileChannel = FileChannel.open(this.filePath, StandardOpenOption.CREATE_NEW, StandardOpenOption.WRITE);
                this.positionInFile = 0L;
                this.lastRotationTime = DateTime.now((DateTimeZone)DateTimeZone.UTC);
            }

            @Override
            public void close() throws IOException {
                this.fileChannel.close();
            }

            public void registerRotationHooks(RotationHooks rotationHooks) {
            }
        }
    }

    private static class QueueEntry {
        static final byte[] ROTATE_FILE_ENTRY = new byte[0];
        static final byte[] FLUSH_FILE_ENTRY = new byte[0];
        private final String topic;
        private final byte[] event;

        QueueEntry(String topic, byte[] event) {
            this.topic = (String)Reject.checkNotNull((Object)topic);
            this.event = (byte[])Reject.checkNotNull((Object)event);
        }

        boolean isRotateEntry() {
            return this.event == ROTATE_FILE_ENTRY;
        }

        boolean isFlushEntry() {
            return this.event == FLUSH_FILE_ENTRY;
        }
    }
}

