/*
 * Decompiled with CFR 0.152.
 */
package io.jans.as.model.crypto;

import com.google.common.collect.Lists;
import com.nimbusds.jose.JOSEException;
import com.nimbusds.jose.JWSAlgorithm;
import com.nimbusds.jose.crypto.impl.ECDSA;
import io.jans.as.model.configuration.AppConfiguration;
import io.jans.as.model.crypto.AbstractCryptoProvider;
import io.jans.as.model.crypto.encryption.KeyEncryptionAlgorithm;
import io.jans.as.model.crypto.signature.AlgorithmFamily;
import io.jans.as.model.crypto.signature.SignatureAlgorithm;
import io.jans.as.model.exception.CryptoProviderException;
import io.jans.as.model.jwk.Algorithm;
import io.jans.as.model.jwk.JSONWebKey;
import io.jans.as.model.jwk.JSONWebKeySet;
import io.jans.as.model.jwk.KeyOpsType;
import io.jans.as.model.jwk.KeySelectionStrategy;
import io.jans.as.model.jwk.Use;
import io.jans.as.model.util.Base64Util;
import io.jans.as.model.util.CertUtils;
import io.jans.as.model.util.Util;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.security.Signature;
import java.security.SignatureException;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.security.interfaces.ECPublicKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.AlgorithmParameterSpec;
import java.security.spec.ECGenParameterSpec;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.UUID;
import java.util.stream.Collectors;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.ASN1EncodableVector;
import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.asn1.DERSequence;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x509.KeyPurposeId;
import org.bouncycastle.cert.CertIOException;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.cert.jcajce.JcaX509CertificateConverter;
import org.bouncycastle.cert.jcajce.JcaX509v3CertificateBuilder;
import org.bouncycastle.jcajce.interfaces.EdDSAPublicKey;
import org.bouncycastle.jcajce.spec.EdDSAParameterSpec;
import org.bouncycastle.operator.ContentSigner;
import org.bouncycastle.operator.OperatorCreationException;
import org.bouncycastle.operator.jcajce.JcaContentSignerBuilder;
import org.json.JSONArray;
import org.json.JSONObject;

public class AuthCryptoProvider
extends AbstractCryptoProvider {
    protected static final Logger LOG = Logger.getLogger(AuthCryptoProvider.class);
    private KeyStore keyStore;
    private String keyStoreFile;
    private String keyStoreSecret;
    private String dnName;
    private final boolean rejectNoneAlg;
    private final KeySelectionStrategy keySelectionStrategy;

    public AuthCryptoProvider() throws KeyStoreException {
        this(null, null, null);
    }

    public AuthCryptoProvider(String keyStoreFile, String keyStoreSecret, String dnName) throws KeyStoreException {
        this(keyStoreFile, keyStoreSecret, dnName, false);
    }

    public AuthCryptoProvider(String keyStoreFile, String keyStoreSecret, String dnName, boolean rejectNoneAlg) throws KeyStoreException {
        this(keyStoreFile, keyStoreSecret, dnName, rejectNoneAlg, AppConfiguration.DEFAULT_KEY_SELECTION_STRATEGY);
    }

    public AuthCryptoProvider(String keyStoreFile, String keyStoreSecret, String dnName, boolean rejectNoneAlg, KeySelectionStrategy keySelectionStrategy) throws KeyStoreException {
        this.rejectNoneAlg = rejectNoneAlg;
        KeySelectionStrategy keySelectionStrategy2 = this.keySelectionStrategy = keySelectionStrategy != null ? keySelectionStrategy : AppConfiguration.DEFAULT_KEY_SELECTION_STRATEGY;
        if (!Util.isNullOrEmpty(keyStoreFile) && !Util.isNullOrEmpty(keyStoreSecret)) {
            this.keyStoreFile = keyStoreFile;
            this.keyStoreSecret = keyStoreSecret;
            this.dnName = dnName;
            this.keyStore = KeyStore.getInstance("PKCS12");
            try {
                File f = new File(keyStoreFile);
                if (!f.exists()) {
                    this.keyStore.load(null, keyStoreSecret.toCharArray());
                    this.store();
                }
                this.load();
            }
            catch (Exception e) {
                LOG.error((Object)e.getMessage(), (Throwable)e);
            }
        }
    }

    private void store() throws IOException, KeyStoreException, NoSuchAlgorithmException, CertificateException {
        try (FileOutputStream fos = new FileOutputStream(this.keyStoreFile);){
            this.keyStore.store(fos, this.keyStoreSecret.toCharArray());
        }
    }

    public void load() throws IOException, NoSuchAlgorithmException, CertificateException {
        try (FileInputStream is = new FileInputStream(this.keyStoreFile);){
            this.keyStore.load(is, this.keyStoreSecret.toCharArray());
            LOG.debug((Object)"Loaded keys from JKS.");
            LOG.trace((Object)("Loaded keys:" + this.getKeys()));
        }
    }

    public String getKeyStoreFile() {
        return this.keyStoreFile;
    }

    public String getKeyStoreSecret() {
        return this.keyStoreSecret;
    }

    public String getDnName() {
        return this.dnName;
    }

    @Override
    public JSONObject generateKey(Algorithm algorithm, Long expirationTime) throws CryptoProviderException {
        return this.generateKey(algorithm, expirationTime, 2048);
    }

    @Override
    public JSONObject generateKey(Algorithm algorithm, Long expirationTime, int keyLength, KeyOpsType keyOpsType) throws CryptoProviderException {
        if (algorithm == null) {
            throw new IllegalArgumentException("The signature algorithm parameter cannot be null");
        }
        JSONObject jsonObject = null;
        try {
            Use algUse = algorithm.getUse();
            if (algUse == Use.SIGNATURE) {
                jsonObject = this.generateKeySignature(algorithm, expirationTime, keyLength, keyOpsType);
            } else if (algUse == Use.ENCRYPTION) {
                jsonObject = this.generateKeyEncryption(algorithm, expirationTime, keyLength, keyOpsType);
            }
        }
        catch (IOException | InvalidAlgorithmParameterException | KeyStoreException | NoSuchAlgorithmException | NoSuchProviderException | CertificateException | OperatorCreationException e) {
            throw new CryptoProviderException(e);
        }
        return jsonObject;
    }

    @Override
    public JSONObject generateKey(Algorithm algorithm, Long expirationTime, int keyLength) throws CryptoProviderException {
        return this.generateKey(algorithm, expirationTime, keyLength, KeyOpsType.CONNECT);
    }

    private static String getKidSuffix(Algorithm algorithm) {
        return "_" + algorithm.getUse().getParamName().toLowerCase() + "_" + algorithm.getParamName().toLowerCase();
    }

    public String getAliasByAlgorithmForDeletion(Algorithm algorithm, String newAlias, KeyOpsType keyOpsType) throws KeyStoreException {
        for (String alias : Collections.list(this.keyStore.aliases())) {
            if (newAlias.equals(alias) || !alias.startsWith(keyOpsType.getValue()) || !alias.endsWith(AuthCryptoProvider.getKidSuffix(algorithm))) continue;
            return alias;
        }
        return null;
    }

    @Override
    public boolean containsKey(String keyId) {
        try {
            if (StringUtils.isBlank((String)keyId)) {
                return false;
            }
            return this.keyStore.getKey(keyId, this.keyStoreSecret.toCharArray()) != null;
        }
        catch (Exception e) {
            LOG.error((Object)e.getMessage(), (Throwable)e);
            return false;
        }
    }

    @Override
    public String sign(String signingInput, String alias, String sharedSecret, SignatureAlgorithm signatureAlgorithm) throws CryptoProviderException {
        try {
            if (signatureAlgorithm == SignatureAlgorithm.NONE) {
                return "";
            }
            if (AlgorithmFamily.HMAC.equals((Object)signatureAlgorithm.getFamily())) {
                SecretKeySpec secretKey = new SecretKeySpec(sharedSecret.getBytes(StandardCharsets.UTF_8), signatureAlgorithm.getAlgorithm());
                Mac mac = Mac.getInstance(signatureAlgorithm.getAlgorithm());
                mac.init(secretKey);
                byte[] sig = mac.doFinal(signingInput.getBytes());
                return Base64Util.base64urlencode(sig);
            }
            PrivateKey privateKey = this.getPrivateKey(alias);
            if (privateKey == null) {
                String error = "Failed to find private key by kid: " + alias + ", signatureAlgorithm: " + signatureAlgorithm + "(check whether web keys JSON in persistence corresponds to keystore file), keySelectionStrategy: " + this.keySelectionStrategy;
                LOG.error((Object)error);
                throw new IllegalStateException(error);
            }
            Signature signer = Signature.getInstance(signatureAlgorithm.getAlgorithm(), "BC");
            signer.initSign(privateKey);
            signer.update(signingInput.getBytes());
            byte[] signature = signer.sign();
            if (AlgorithmFamily.EC.equals((Object)signatureAlgorithm.getFamily())) {
                int signatureLenght = ECDSA.getSignatureByteArrayLength((JWSAlgorithm)signatureAlgorithm.getJwsAlgorithm());
                signature = ECDSA.transcodeSignatureToConcat((byte[])signature, (int)signatureLenght);
            }
            return Base64Util.base64urlencode(signature);
        }
        catch (JOSEException | InvalidKeyException | NoSuchAlgorithmException | NoSuchProviderException | SignatureException e) {
            throw new CryptoProviderException(e);
        }
    }

    @Override
    public boolean verifySignature(String signingInput, String encodedSignature, String alias, JSONObject jwks, String sharedSecret, SignatureAlgorithm signatureAlgorithm) throws CryptoProviderException {
        if (this.rejectNoneAlg && signatureAlgorithm == SignatureAlgorithm.NONE) {
            LOG.trace((Object)"None algorithm is forbidden by `rejectJwtWithNoneAlg` property.");
            return false;
        }
        if (signatureAlgorithm == SignatureAlgorithm.NONE) {
            return Util.isNullOrEmpty(encodedSignature);
        }
        if (AlgorithmFamily.HMAC.equals((Object)signatureAlgorithm.getFamily())) {
            String expectedSignature = this.sign(signingInput, null, sharedSecret, signatureAlgorithm);
            return expectedSignature.equals(encodedSignature);
        }
        return this.verifySignatureEcEdRSA(signingInput, encodedSignature, alias, jwks, signatureAlgorithm);
    }

    @Override
    public boolean deleteKey(String alias) throws CryptoProviderException {
        try {
            this.keyStore.deleteEntry(alias);
        }
        catch (KeyStoreException e) {
            throw new CryptoProviderException(e);
        }
        try (FileOutputStream stream = new FileOutputStream(this.keyStoreFile);){
            this.keyStore.store(stream, this.keyStoreSecret.toCharArray());
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            throw new CryptoProviderException(e);
        }
        return true;
    }

    @Override
    public PublicKey getPublicKey(String alias) throws CryptoProviderException {
        if (Util.isNullOrEmpty(alias) || this.keyStore == null) {
            return null;
        }
        try {
            Certificate certificate = this.keyStore.getCertificate(alias);
            if (certificate == null) {
                return null;
            }
            this.checkKeyExpiration(alias);
            return certificate.getPublicKey();
        }
        catch (KeyStoreException e) {
            throw new CryptoProviderException(e);
        }
    }

    @Override
    public String getKeyId(JSONWebKeySet jsonWebKeySet, Algorithm algorithm, Use use, KeyOpsType keyOpsType) throws CryptoProviderException {
        if (algorithm == null || AlgorithmFamily.HMAC.equals((Object)algorithm.getFamily())) {
            return null;
        }
        try {
            String kid = null;
            List<JSONWebKey> keys = jsonWebKeySet.getKeys();
            LOG.trace((Object)("WebKeys:" + keys.stream().map(JSONWebKey::getKid).collect(Collectors.toList())));
            LOG.trace((Object)("KeyStoreKeys:" + this.getKeys()));
            ArrayList<JSONWebKey> keysByAlgAndUse = new ArrayList<JSONWebKey>();
            for (JSONWebKey key : keys) {
                Key keyFromStore;
                boolean keyOpsCondition;
                boolean bl = keyOpsCondition = keyOpsType == null || key.getKeyOpsType() == null || key.getKeyOpsType().isEmpty() || key.getKeyOpsType().contains((Object)keyOpsType);
                if (algorithm != key.getAlg() || use != null && use != key.getUse() || !keyOpsCondition || (keyFromStore = this.keyStore.getKey(kid = key.getKid(), this.keyStoreSecret.toCharArray())) == null) continue;
                keysByAlgAndUse.add(key);
            }
            if (keysByAlgAndUse.isEmpty()) {
                LOG.trace((Object)("kid is not in keystore, algorithm: {}" + algorithm + ", kid: " + kid + ", keyStorePath:" + this.keyStoreFile + ", keyOpsType: " + keyOpsType + ", use: " + use));
                return kid;
            }
            JSONWebKey selectedKey = this.keySelectionStrategy.select(keysByAlgAndUse);
            String selectedKid = selectedKey != null ? selectedKey.getKid() : null;
            LOG.trace((Object)("Selected kid: " + selectedKid + ", keySelection Strategy: " + this.keySelectionStrategy));
            return selectedKid;
        }
        catch (KeyStoreException | NoSuchAlgorithmException | UnrecoverableKeyException e) {
            throw new CryptoProviderException(e);
        }
    }

    @Override
    public PrivateKey getPrivateKey(String alias) throws CryptoProviderException {
        if (Util.isNullOrEmpty(alias)) {
            return null;
        }
        try {
            Key key = this.keyStore.getKey(alias, this.keyStoreSecret.toCharArray());
            if (key == null) {
                return null;
            }
            PrivateKey privateKey = (PrivateKey)key;
            this.checkKeyExpiration(alias);
            return privateKey;
        }
        catch (KeyStoreException | NoSuchAlgorithmException | UnrecoverableKeyException e) {
            throw new CryptoProviderException(e);
        }
    }

    public X509Certificate generateV3Certificate(KeyPair keyPair, String issuer, String signatureAlgorithm, Long expirationTime) throws CertIOException, OperatorCreationException, CertificateException {
        PrivateKey privateKey = keyPair.getPrivate();
        PublicKey publicKey = keyPair.getPublic();
        X500Name issuerName = new X500Name(issuer);
        X500Name subjectName = new X500Name(issuer);
        BigInteger serial = new BigInteger(256, new SecureRandom());
        Date notBefore = new Date(System.currentTimeMillis() - 10000L);
        Date notAfter = new Date(expirationTime);
        JcaX509v3CertificateBuilder builder = new JcaX509v3CertificateBuilder(issuerName, serial, notBefore, notAfter, subjectName, publicKey);
        ASN1EncodableVector purposes = new ASN1EncodableVector();
        purposes.add((ASN1Encodable)KeyPurposeId.id_kp_serverAuth);
        purposes.add((ASN1Encodable)KeyPurposeId.id_kp_clientAuth);
        purposes.add((ASN1Encodable)KeyPurposeId.anyExtendedKeyUsage);
        ASN1ObjectIdentifier extendedKeyUsage = new ASN1ObjectIdentifier("2.5.29.37").intern();
        builder.addExtension(extendedKeyUsage, false, (ASN1Encodable)new DERSequence(purposes));
        ContentSigner signer = new JcaContentSignerBuilder(signatureAlgorithm).setProvider("BC").build(privateKey);
        X509CertificateHolder holder = builder.build(signer);
        return new JcaX509CertificateConverter().setProvider("BC").getCertificate(holder);
    }

    @Override
    public List<String> getKeys() {
        try {
            return Collections.list(this.keyStore.aliases());
        }
        catch (KeyStoreException e) {
            LOG.error((Object)e.getMessage(), (Throwable)e);
            return Lists.newArrayList();
        }
    }

    public SignatureAlgorithm getSignatureAlgorithm(String alias) throws KeyStoreException {
        Certificate[] chain = this.keyStore.getCertificateChain(alias);
        if (chain == null || chain.length == 0) {
            return null;
        }
        X509Certificate cert = (X509Certificate)chain[0];
        return CertUtils.getSignatureAlgorithm(cert);
    }

    private void checkKeyExpiration(String alias) {
        try {
            Date expirationDate = ((X509Certificate)this.keyStore.getCertificate(alias)).getNotAfter();
            this.checkKeyExpiration(alias, expirationDate.getTime());
        }
        catch (KeyStoreException e) {
            e.printStackTrace();
        }
    }

    public KeyStore getKeyStore() {
        return this.keyStore;
    }

    private JSONObject generateKeySignature(Algorithm algorithm, Long expirationTime, int keyLength, KeyOpsType keyOpsType) throws NoSuchAlgorithmException, NoSuchProviderException, InvalidAlgorithmParameterException, OperatorCreationException, CertificateException, KeyStoreException, IOException {
        SignatureAlgorithm signatureAlgorithm = SignatureAlgorithm.fromString(algorithm.getParamName());
        if (signatureAlgorithm == null) {
            algorithm = Algorithm.ES384;
            signatureAlgorithm = SignatureAlgorithm.ES384;
        }
        KeyPairGenerator keyGen = null;
        AlgorithmFamily algorithmFamily = algorithm.getFamily();
        switch (algorithmFamily) {
            case RSA: {
                keyGen = KeyPairGenerator.getInstance(algorithmFamily.toString(), "BC");
                keyGen.initialize(keyLength, new SecureRandom());
                break;
            }
            case EC: {
                ECGenParameterSpec eccgen = new ECGenParameterSpec(signatureAlgorithm.getCurve().getAlias());
                keyGen = KeyPairGenerator.getInstance(algorithmFamily.toString(), "BC");
                keyGen.initialize(eccgen, new SecureRandom());
                break;
            }
            case ED: {
                EdDSAParameterSpec edSpec = new EdDSAParameterSpec(signatureAlgorithm.getCurve().getAlias());
                keyGen = KeyPairGenerator.getInstance(signatureAlgorithm.getName(), "BC");
                keyGen.initialize((AlgorithmParameterSpec)edSpec, new SecureRandom());
                break;
            }
            default: {
                throw new IllegalStateException("The provided signature algorithm parameter is not supported: algorithmFamily = " + algorithmFamily);
            }
        }
        return this.getJson(algorithm, keyGen, signatureAlgorithm.getAlgorithm(), expirationTime, keyOpsType);
    }

    private JSONObject generateKeyEncryption(Algorithm algorithm, Long expirationTime, int keyLength, KeyOpsType keyOpsType) throws NoSuchAlgorithmException, NoSuchProviderException, InvalidAlgorithmParameterException, OperatorCreationException, CertificateException, KeyStoreException, IOException {
        KeyEncryptionAlgorithm keyEncryptionAlgorithm = KeyEncryptionAlgorithm.fromName(algorithm.getParamName());
        if (keyEncryptionAlgorithm == null) {
            algorithm = Algorithm.RS256;
            keyEncryptionAlgorithm = KeyEncryptionAlgorithm.RSA1_5;
        }
        KeyPairGenerator keyGen = null;
        String signatureAlgorithm = null;
        AlgorithmFamily algorithmFamily = algorithm.getFamily();
        switch (algorithmFamily) {
            case RSA: {
                keyGen = KeyPairGenerator.getInstance(algorithmFamily.toString(), "BC");
                keyGen.initialize(keyLength, new SecureRandom());
                signatureAlgorithm = "SHA256WITHRSA";
                break;
            }
            case EC: {
                ECGenParameterSpec eccgen = new ECGenParameterSpec(keyEncryptionAlgorithm.getCurve().getAlias());
                keyGen = KeyPairGenerator.getInstance(algorithmFamily.toString(), "BC");
                keyGen.initialize(eccgen, new SecureRandom());
                signatureAlgorithm = "SHA256WITHECDSA";
                break;
            }
            default: {
                throw new IllegalStateException("The provided key encryption algorithm parameter is not supported: algorithmFamily = " + algorithmFamily);
            }
        }
        return this.getJson(algorithm, keyGen, signatureAlgorithm, expirationTime, keyOpsType);
    }

    private String getKid(Algorithm algorithm, KeyOpsType keyOpsType) {
        if (keyOpsType == null) {
            keyOpsType = KeyOpsType.CONNECT;
        }
        return keyOpsType.getValue() + "_" + UUID.randomUUID().toString() + AuthCryptoProvider.getKidSuffix(algorithm);
    }

    private JSONObject getJson(Algorithm algorithm, KeyPairGenerator keyGen, String signatureAlgorithmStr, Long expirationTime, KeyOpsType keyOpsType) throws NoSuchAlgorithmException, OperatorCreationException, CertificateException, KeyStoreException, IOException {
        KeyPair keyPair = keyGen.generateKeyPair();
        PrivateKey pk = keyPair.getPrivate();
        X509Certificate cert = this.generateV3Certificate(keyPair, this.dnName, signatureAlgorithmStr, expirationTime);
        Certificate[] chain = new X509Certificate[]{cert};
        String alias = this.getKid(algorithm, keyOpsType);
        this.keyStore.setKeyEntry(alias, pk, this.keyStoreSecret.toCharArray(), chain);
        String oldAliasByAlgorithm = this.getAliasByAlgorithmForDeletion(algorithm, alias, keyOpsType);
        if (StringUtils.isNotBlank((String)oldAliasByAlgorithm)) {
            this.keyStore.deleteEntry(oldAliasByAlgorithm);
            LOG.trace((Object)("New key: " + alias + ", deleted key: " + oldAliasByAlgorithm));
        }
        try (FileOutputStream stream = new FileOutputStream(this.keyStoreFile);){
            this.keyStore.store(stream, this.keyStoreSecret.toCharArray());
        }
        PublicKey publicKey = keyPair.getPublic();
        Use use = algorithm.getUse();
        JSONObject jsonObject = new JSONObject();
        algorithm.fill(jsonObject);
        jsonObject.put("kid", (Object)alias);
        jsonObject.put("exp", (Object)expirationTime);
        if (publicKey instanceof RSAPublicKey) {
            RSAPublicKey rsaPublicKey = (RSAPublicKey)publicKey;
            jsonObject.put("n", (Object)Base64Util.base64urlencodeUnsignedBigInt(rsaPublicKey.getModulus()));
            jsonObject.put("e", (Object)Base64Util.base64urlencodeUnsignedBigInt(rsaPublicKey.getPublicExponent()));
        } else if (publicKey instanceof ECPublicKey) {
            ECPublicKey ecPublicKey = (ECPublicKey)publicKey;
            if (use == Use.SIGNATURE) {
                SignatureAlgorithm signatureAlgorithm = SignatureAlgorithm.fromString(algorithm.getParamName());
                jsonObject.put("crv", (Object)signatureAlgorithm.getCurve().getName());
            } else if (use == Use.ENCRYPTION) {
                KeyEncryptionAlgorithm keyEncryptionAlgorithm = KeyEncryptionAlgorithm.fromName(algorithm.getParamName());
                jsonObject.put("crv", (Object)keyEncryptionAlgorithm.getCurve().getName());
            }
            jsonObject.put("x", (Object)Base64Util.base64urlencodeUnsignedBigInt(ecPublicKey.getW().getAffineX()));
            jsonObject.put("y", (Object)Base64Util.base64urlencodeUnsignedBigInt(ecPublicKey.getW().getAffineY()));
        } else if (use == Use.SIGNATURE && publicKey instanceof EdDSAPublicKey) {
            EdDSAPublicKey edDSAPublicKey = (EdDSAPublicKey)publicKey;
            SignatureAlgorithm signatureAlgorithm = SignatureAlgorithm.fromString(algorithm.getParamName());
            jsonObject.put("crv", (Object)signatureAlgorithm.getCurve().getName());
            jsonObject.put("x", (Object)Base64Util.base64urlencode(edDSAPublicKey.getEncoded()));
        }
        JSONArray x5c = new JSONArray();
        x5c.put((Object)Base64.encodeBase64String((byte[])cert.getEncoded()));
        jsonObject.put("x5c", (Object)x5c);
        return jsonObject;
    }

    private boolean verifySignatureEcEdRSA(String signingInput, String encodedSignature, String alias, JSONObject jwks, SignatureAlgorithm signatureAlgorithm) {
        PublicKey publicKey = null;
        try {
            publicKey = jwks == null ? this.getPublicKey(alias) : this.getPublicKey(alias, jwks, signatureAlgorithm.getAlg());
            if (publicKey == null) {
                return false;
            }
            return this.verifySignatureEcEdRSA(signingInput, encodedSignature, signatureAlgorithm, publicKey);
        }
        catch (Exception e) {
            LOG.error((Object)e.getMessage(), (Throwable)e);
            return false;
        }
    }

    private boolean verifySignatureEcEdRSA(String signingInput, String encodedSignature, SignatureAlgorithm signatureAlgorithm, PublicKey publicKey) throws JOSEException, NoSuchAlgorithmException, NoSuchProviderException, InvalidKeyException, SignatureException {
        byte[] signature;
        byte[] signatureDer = signature = Base64Util.base64urldecode(encodedSignature);
        if (AlgorithmFamily.EC.equals((Object)signatureAlgorithm.getFamily())) {
            signatureDer = ECDSA.transcodeSignatureToDER((byte[])signatureDer);
        }
        Signature verifier = Signature.getInstance(signatureAlgorithm.getAlgorithm(), "BC");
        verifier.initVerify(publicKey);
        verifier.update(signingInput.getBytes());
        try {
            return verifier.verify(signatureDer);
        }
        catch (SignatureException e) {
            return verifier.verify(signature);
        }
    }
}

