/*
 * Decompiled with CFR 0.152.
 */
package io.jans.as.model.crypto;

import com.google.common.collect.Lists;
import io.jans.as.model.configuration.AppConfiguration;
import io.jans.as.model.crypto.signature.AlgorithmFamily;
import io.jans.as.model.crypto.signature.EllipticEdvardsCurve;
import io.jans.as.model.crypto.signature.SignatureAlgorithm;
import io.jans.as.model.exception.CryptoProviderException;
import io.jans.as.model.exception.InvalidParameterException;
import io.jans.as.model.jwk.Algorithm;
import io.jans.as.model.jwk.JSONWebKey;
import io.jans.as.model.jwk.JSONWebKeySet;
import io.jans.as.model.jwk.KeyOpsType;
import io.jans.as.model.jwk.Use;
import io.jans.as.model.util.Base64Util;
import io.jans.eleven.model.JwksRequestParam;
import io.jans.eleven.model.KeyRequestParam;
import java.math.BigInteger;
import java.security.AlgorithmParameters;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.spec.ECGenParameterSpec;
import java.security.spec.ECParameterSpec;
import java.security.spec.ECPoint;
import java.security.spec.ECPublicKeySpec;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.InvalidParameterSpecException;
import java.security.spec.RSAPublicKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.TimeZone;
import org.apache.log4j.Logger;
import org.json.JSONArray;
import org.json.JSONObject;

public abstract class AbstractCryptoProvider {
    protected static final Logger LOG = Logger.getLogger(AbstractCryptoProvider.class);
    private static final String DEF_EXPIRESON = "\n\tExpires On: ";
    private static final String DEF_TODAYSDATE = "\n\tToday's Date: ";
    private static final String DEF_DAYS = " days";
    private int keyRegenerationIntervalInDays = -1;

    public abstract JSONObject generateKey(Algorithm var1, Long var2) throws CryptoProviderException;

    public abstract JSONObject generateKey(Algorithm var1, Long var2, int var3) throws CryptoProviderException;

    public abstract JSONObject generateKey(Algorithm var1, Long var2, int var3, KeyOpsType var4) throws CryptoProviderException;

    public abstract String sign(String var1, String var2, String var3, SignatureAlgorithm var4) throws CryptoProviderException;

    public abstract boolean verifySignature(String var1, String var2, String var3, JSONObject var4, String var5, SignatureAlgorithm var6) throws CryptoProviderException;

    public abstract boolean deleteKey(String var1) throws CryptoProviderException;

    public abstract boolean containsKey(String var1);

    public List<String> getKeys() {
        return Lists.newArrayList();
    }

    public abstract PrivateKey getPrivateKey(String var1) throws CryptoProviderException;

    public abstract PublicKey getPublicKey(String var1) throws CryptoProviderException;

    public String getKeyId(JSONWebKeySet jsonWebKeySet, Algorithm algorithm, Use use, KeyOpsType keyOps) throws CryptoProviderException {
        if (algorithm == null || AlgorithmFamily.HMAC.equals((Object)algorithm.getFamily())) {
            return null;
        }
        for (JSONWebKey key : jsonWebKeySet.getKeys()) {
            if (algorithm != key.getAlg() || use != null && use != key.getUse()) continue;
            return key.getKid();
        }
        return null;
    }

    public JwksRequestParam getJwksRequestParam(JSONObject jwkJsonObject) {
        JwksRequestParam jwks = new JwksRequestParam();
        jwks.setKeyRequestParams(new ArrayList());
        KeyRequestParam key = new KeyRequestParam();
        key.setAlg(jwkJsonObject.getString("alg"));
        key.setKid(jwkJsonObject.getString("kid"));
        key.setUse(jwkJsonObject.getString("use"));
        key.setKty(jwkJsonObject.getString("kty"));
        key.setN(jwkJsonObject.optString("n"));
        key.setE(jwkJsonObject.optString("e"));
        key.setCrv(jwkJsonObject.optString("crv"));
        key.setX(jwkJsonObject.optString("x"));
        key.setY(jwkJsonObject.optString("y"));
        jwks.getKeyRequestParams().add(key);
        return jwks;
    }

    public static JSONObject generateJwks(AbstractCryptoProvider cryptoProvider, AppConfiguration configuration) {
        GregorianCalendar expirationTime = new GregorianCalendar(TimeZone.getTimeZone("UTC"));
        expirationTime.add(10, configuration.getKeyRegenerationInterval());
        expirationTime.add(13, configuration.getIdTokenLifetime());
        long expiration = expirationTime.getTimeInMillis();
        List<String> allowedAlgs = configuration.getKeyAlgsAllowedForGeneration();
        JSONArray keys = new JSONArray();
        for (Algorithm alg : Algorithm.values()) {
            try {
                boolean isNotSupported;
                boolean isNotAllowed = !allowedAlgs.isEmpty() && !allowedAlgs.contains(alg.getParamName());
                boolean bl = isNotSupported = !alg.canGenerateKeys();
                if (isNotAllowed || isNotSupported) {
                    if (isNotAllowed) {
                        LOG.debug((Object)String.format("Key generation for %s is skipped because it's not allowed by keyAlgsAllowedForGeneration configuration property.", alg.toString()));
                    }
                    if (!isNotSupported) continue;
                    LOG.trace((Object)(alg + " does not support keys re-generation."));
                    continue;
                }
                keys.put((Object)cryptoProvider.generateKey(alg, expiration));
            }
            catch (Exception ex) {
                LOG.error((Object)String.format("Algorithm: %s", new Object[]{alg}), (Throwable)ex);
            }
        }
        JSONObject jsonObject = new JSONObject();
        jsonObject.put("keys", (Object)keys);
        return jsonObject;
    }

    public PublicKey getPublicKey(String alias, JSONObject jwks, Algorithm requestedAlgorithm) throws CryptoProviderException {
        JSONArray webKeys = jwks.getJSONArray("keys");
        try {
            if (alias == null) {
                if (webKeys.length() == 1) {
                    JSONObject key = webKeys.getJSONObject(0);
                    return this.processKey(requestedAlgorithm, alias, key);
                }
                return null;
            }
            for (int i = 0; i < webKeys.length(); ++i) {
                PublicKey publicKey;
                JSONObject key = webKeys.getJSONObject(i);
                if (!alias.equals(key.getString("kid")) || (publicKey = this.processKey(requestedAlgorithm, alias, key)) == null) continue;
                return publicKey;
            }
        }
        catch (InvalidParameterException | NoSuchAlgorithmException | InvalidKeySpecException | InvalidParameterSpecException e) {
            throw new CryptoProviderException(e);
        }
        return null;
    }

    private PublicKey processKey(Algorithm requestedAlgorithm, String alias, JSONObject key) throws NoSuchAlgorithmException, InvalidKeySpecException, InvalidParameterSpecException, InvalidParameterException {
        PublicKey publicKey = null;
        AlgorithmFamily algorithmFamily = null;
        if (key.has("alg")) {
            Algorithm algorithm = Algorithm.fromString(key.optString("alg"));
            if (requestedAlgorithm != null && !requestedAlgorithm.equals((Object)algorithm)) {
                LOG.trace((Object)("kid matched but algorithm does not match. kid algorithm:" + algorithm + ", requestedAlgorithm:" + requestedAlgorithm + ", kid:" + alias));
                return null;
            }
            algorithmFamily = algorithm.getFamily();
        } else if (key.has("kty")) {
            algorithmFamily = AlgorithmFamily.fromString(key.getString("kty"));
        } else {
            throw new InvalidParameterException("Wrong key (JSONObject): doesn't contain 'alg' and 'kty' properties");
        }
        switch (algorithmFamily) {
            case RSA: {
                KeyFactory keyFactory = KeyFactory.getInstance("RSA");
                RSAPublicKeySpec pubKeySpec = new RSAPublicKeySpec(new BigInteger(1, Base64Util.base64urldecode(key.getString("n"))), new BigInteger(1, Base64Util.base64urldecode(key.getString("e"))));
                publicKey = keyFactory.generatePublic(pubKeySpec);
                break;
            }
            case EC: {
                EllipticEdvardsCurve curve = EllipticEdvardsCurve.fromString(key.optString("crv"));
                AlgorithmParameters parameters = AlgorithmParameters.getInstance(AlgorithmFamily.EC.toString());
                parameters.init(new ECGenParameterSpec(curve.getAlias()));
                ECParameterSpec ecParameters = parameters.getParameterSpec(ECParameterSpec.class);
                publicKey = KeyFactory.getInstance(AlgorithmFamily.EC.toString()).generatePublic(new ECPublicKeySpec(new ECPoint(new BigInteger(1, Base64Util.base64urldecode(key.getString("x"))), new BigInteger(1, Base64Util.base64urldecode(key.getString("y")))), ecParameters));
                break;
            }
            case ED: {
                X509EncodedKeySpec publicKeySpec = new X509EncodedKeySpec(Base64Util.base64urldecode(key.getString("x")));
                publicKey = KeyFactory.getInstance(key.optString("alg")).generatePublic(publicKeySpec);
                break;
            }
            default: {
                throw new InvalidParameterException(String.format("Wrong AlgorithmFamily value: %s", new Object[]{algorithmFamily}));
            }
        }
        if (key.has("exp")) {
            this.checkKeyExpiration(alias, key.getLong("exp"));
        }
        return publicKey;
    }

    protected void checkKeyExpiration(String alias, Long expirationTime) {
        try {
            Date expirationDate = new Date(expirationTime);
            SimpleDateFormat ft = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
            Date today = new Date();
            long expiresInDays = (expirationTime - today.getTime()) / 86400000L;
            if (expiresInDays == 0L) {
                LOG.warn((Object)("\nWARNING! Key will expire soon, alias: " + alias + DEF_EXPIRESON + ft.format(expirationDate) + DEF_TODAYSDATE + ft.format(today)));
                return;
            }
            if (expiresInDays < 0L) {
                LOG.warn((Object)("\nWARNING! Expired Key is used, alias: " + alias + DEF_EXPIRESON + ft.format(expirationDate) + DEF_TODAYSDATE + ft.format(today)));
                return;
            }
            if (this.keyRegenerationIntervalInDays <= 0 && expiresInDays < 30L) {
                LOG.warn((Object)("\nWARNING! Key with alias: " + alias + "\n\tExpires In: " + expiresInDays + " days\n\tExpires On: " + ft.format(expirationDate) + DEF_TODAYSDATE + ft.format(today)));
                return;
            }
            if (expiresInDays < (long)this.keyRegenerationIntervalInDays) {
                LOG.warn((Object)("\nWARNING! Key with alias: " + alias + "\n\tExpires In: " + expiresInDays + " days\n\tExpires On: " + ft.format(expirationDate) + "\n\tKey Regeneration In: " + this.keyRegenerationIntervalInDays + " days\n\tToday's Date: " + ft.format(today)));
            }
        }
        catch (Exception e) {
            LOG.error((Object)"Failed to check key expiration.", (Throwable)e);
        }
    }

    public int getKeyRegenerationIntervalInDays() {
        return this.keyRegenerationIntervalInDays;
    }

    public void setKeyRegenerationIntervalInDays(int keyRegenerationIntervalInDays) {
        this.keyRegenerationIntervalInDays = keyRegenerationIntervalInDays;
    }
}

